/*
 * Decompiled with CFR 0.152.
 */
package net.starlark.java.lib.json;

import java.util.Arrays;
import java.util.Map;
import net.starlark.java.annot.Param;
import net.starlark.java.annot.StarlarkBuiltin;
import net.starlark.java.annot.StarlarkMethod;
import net.starlark.java.eval.Dict;
import net.starlark.java.eval.EvalException;
import net.starlark.java.eval.Mutability;
import net.starlark.java.eval.Starlark;
import net.starlark.java.eval.StarlarkFloat;
import net.starlark.java.eval.StarlarkInt;
import net.starlark.java.eval.StarlarkIterable;
import net.starlark.java.eval.StarlarkList;
import net.starlark.java.eval.StarlarkThread;
import net.starlark.java.eval.StarlarkValue;
import net.starlark.java.eval.Structure;

@StarlarkBuiltin(name="json", category="core.lib", doc="Module json is a Starlark module of JSON-related functions.")
public final class Json
implements StarlarkValue {
    public static final Json INSTANCE = new Json();
    private static final char[] HEX = "0123456789abcdef".toCharArray();

    private Json() {
    }

    @StarlarkMethod(name="encode", doc="<p>The encode function accepts one required positional argument, which it converts to JSON by cases:\n<ul>\n<li>None, True, and False are converted to 'null', 'true', and 'false', respectively.\n<li>An int, no matter how large, is encoded as a decimal integer. Some decoders may not be able to decode very large integers.\n<li>A float is encoded using a decimal point or an exponent or both, even if its numeric value is an integer. It is an error to encode a non-finite  floating-point value.\n<li>A string value is encoded as a JSON string literal that denotes the value.  Each unpaired surrogate is replaced by U+FFFD.\n<li>A dict is encoded as a JSON object, in key order.  It is an error if any key is not a string.\n<li>A list or tuple is encoded as a JSON array.\n<li>A struct-like value is encoded as a JSON object, in field name order.\n</ul>\nAn application-defined type may define its own JSON encoding.\nEncoding any other value yields an error.\n", parameters={@Param(name="x")})
    public String encode(Object x) throws EvalException {
        Encoder enc = new Encoder();
        try {
            enc.encode(x);
        }
        catch (StackOverflowError unused) {
            throw Starlark.errorf("nesting depth limit exceeded", new Object[0]);
        }
        return enc.out.toString();
    }

    @StarlarkMethod(name="decode", doc="The decode function has one required positional parameter: a JSON string.\nIt returns the Starlark value that the string denotes.\n<ul><li><code>\"null\"</code>, <code>\"true\"</code> and <code>\"false\"</code> are parsed as <code>None</code>, <code>True</code>, and <code>False</code>.\n<li>Numbers are parsed as int, or as a float if they contain a decimal point or an exponent. Although JSON has no syntax  for non-finite values, very large values may be decoded as infinity.\n<li>a JSON object is parsed as a new unfrozen Starlark dict. If the same key string occurs more than once in the object, the last value for the key is kept.\n<li>a JSON array is parsed as new unfrozen Starlark list.\n</ul>\nIf <code>x</code> is not a valid JSON encoding and the optional, positional-only <code>default</code> parameter is specified (including specified as <code>None</code>), this function returns the <code>default</code> value.\nIf <code>x</code> is not a valid JSON encoding and the optional <code>default</code> parameter is <em>not</em> specified, this function fails.", parameters={@Param(name="x", doc="JSON string to decode."), @Param(name="default", doc="If specified, the value to return when <code>x</code> cannot be decoded.", defaultValue="unbound")}, useStarlarkThread=true)
    public Object decode(String x, Object defaultValue, StarlarkThread thread) throws EvalException {
        try {
            return new Decoder(thread.mutability(), x).decode();
        }
        catch (EvalException e) {
            if (defaultValue != Starlark.UNBOUND) {
                return defaultValue;
            }
            throw e;
        }
    }

    @StarlarkMethod(name="indent", doc="The indent function returns the indented form of a valid JSON-encoded string.\nEach array element or object field appears on a new line, beginning with the prefix string followed by one or more copies of the indent string, according to its nesting depth.\nThe function accepts one required positional parameter, the JSON string,\nand two optional keyword-only string parameters, prefix and indent,\nthat specify a prefix of each new line, and the unit of indentation.\nIf the input is not valid, the function may fail or return invalid output.\n", parameters={@Param(name="s"), @Param(name="prefix", positional=false, named=true, defaultValue="''"), @Param(name="indent", positional=false, named=true, defaultValue="'\\t'")})
    public String indent(String s2, String prefix, String indent) throws EvalException {
        Indenter in = new Indenter(prefix, indent, s2);
        try {
            in.indent();
        }
        catch (StringIndexOutOfBoundsException unused) {
            throw Starlark.errorf("input is not valid JSON", new Object[0]);
        }
        return in.out.toString();
    }

    @StarlarkMethod(name="encode_indent", doc="The encode_indent function is equivalent to <code>json.indent(json.encode(x), ...)</code>. See <code>indent</code> for description of formatting parameters.", parameters={@Param(name="x"), @Param(name="prefix", positional=false, named=true, defaultValue="''"), @Param(name="indent", positional=false, named=true, defaultValue="'\\t'")})
    public String encodeIndent(Object x, String prefix, String indent) throws EvalException {
        return this.indent(this.encode(x), prefix, indent);
    }

    private static boolean isdigit(char c) {
        return c >= '0' && c <= '9';
    }

    private static String quoteChar(char c) {
        return Starlark.repr("" + c);
    }

    private static final class Indenter {
        private final StringBuilder out = new StringBuilder();
        private final String prefix;
        private final String indent;
        private final String s;
        private int i;

        Indenter(String prefix, String indent, String s2) {
            this.prefix = prefix;
            this.indent = indent;
            this.s = s2;
        }

        private void indent() throws EvalException {
            int depth = 0;
            do {
                char c = this.next();
                int start = this.i++;
                switch (c) {
                    case '\"': {
                        c = this.s.charAt(this.i);
                        while (c != '\"') {
                            if (c == '\\' && (c = this.s.charAt(++this.i)) == 'u') {
                                this.i += 4;
                            }
                            c = this.s.charAt(++this.i);
                        }
                        ++this.i;
                        this.out.append(this.s, start, this.i);
                        break;
                    }
                    case 'n': {
                        this.i += "null".length();
                        this.out.append(this.s, start, this.i);
                        break;
                    }
                    case 't': {
                        this.i += "true".length();
                        this.out.append(this.s, start, this.i);
                        break;
                    }
                    case 'f': {
                        this.i += "false".length();
                        this.out.append(this.s, start, this.i);
                        break;
                    }
                    case ',': {
                        ++this.i;
                        this.out.append(',');
                        this.newline(depth);
                        break;
                    }
                    case '[': 
                    case '{': {
                        ++this.i;
                        this.out.append(c);
                        c = this.next();
                        if (c == ']' || c == '}') {
                            ++this.i;
                            this.out.append(c);
                            break;
                        }
                        this.newline(++depth);
                        break;
                    }
                    case ']': 
                    case '}': {
                        ++this.i;
                        this.newline(--depth);
                        this.out.append(c);
                        break;
                    }
                    case ':': {
                        ++this.i;
                        this.out.append(": ");
                        break;
                    }
                    default: {
                        if (!Json.isdigit(c) && c != '-') {
                            throw Starlark.errorf("unexpected character %s", Json.quoteChar(c));
                        }
                        while (this.i < this.s.length() && (Json.isdigit(c = this.s.charAt(++this.i)) || c == '.' || c == 'e' || c == 'E' || c == '+' || c == '-')) {
                        }
                        this.out.append(this.s, start, this.i);
                    }
                }
            } while (depth > 0);
        }

        private void newline(int depth) {
            this.out.append('\n').append(this.prefix);
            for (int i = 0; i < depth; ++i) {
                this.out.append(this.indent);
            }
        }

        private boolean skipSpace() {
            while (this.i < this.s.length()) {
                char c = this.s.charAt(this.i);
                if (c != ' ' && c != '\t' && c != '\n' && c != '\r') {
                    return true;
                }
                ++this.i;
            }
            return false;
        }

        private char next() throws EvalException {
            if (this.skipSpace()) {
                return this.s.charAt(this.i);
            }
            throw Starlark.errorf("unexpected end of file", new Object[0]);
        }
    }

    private static final class Decoder {
        private final Mutability mu;
        private final String s;
        private int i = 0;

        private Decoder(Mutability mu, String s2) {
            this.mu = mu;
            this.s = s2;
        }

        private Object decode() throws EvalException {
            try {
                Object x = this.parse();
                if (this.skipSpace()) {
                    throw Starlark.errorf("unexpected character %s after value", Json.quoteChar(this.s.charAt(this.i)));
                }
                return x;
            }
            catch (StackOverflowError unused) {
                throw Starlark.errorf("nesting depth limit exceeded", new Object[0]);
            }
            catch (EvalException ex) {
                throw Starlark.errorf("at offset %d, %s", this.i, ex.getMessage());
            }
        }

        private Object parse() throws EvalException {
            char c = this.next();
            switch (c) {
                case '\"': {
                    return this.parseString();
                }
                case 'n': {
                    if (!this.s.startsWith("null", this.i)) break;
                    this.i += "null".length();
                    return Starlark.NONE;
                }
                case 't': {
                    if (!this.s.startsWith("true", this.i)) break;
                    this.i += "true".length();
                    return true;
                }
                case 'f': {
                    if (!this.s.startsWith("false", this.i)) break;
                    this.i += "false".length();
                    return false;
                }
                case '[': {
                    StarlarkList list = StarlarkList.newList(this.mu);
                    ++this.i;
                    c = this.next();
                    if (c != ']') {
                        while (true) {
                            Object elem = this.parse();
                            list.addElement(elem);
                            c = this.next();
                            if (c != ',') {
                                if (c == ']') break;
                                throw Starlark.errorf("got %s, want ',' or ']'", Json.quoteChar(c));
                            }
                            ++this.i;
                        }
                    }
                    ++this.i;
                    return list;
                }
                case '{': {
                    Dict<String, Object> dict = Dict.of(this.mu);
                    ++this.i;
                    c = this.next();
                    if (c != '}') {
                        while (true) {
                            Object key;
                            if (!((key = this.parse()) instanceof String)) {
                                throw Starlark.errorf("got %s for object key, want string", Starlark.type(key));
                            }
                            c = this.next();
                            if (c != ':') {
                                throw Starlark.errorf("after object key, got %s, want ':' ", Json.quoteChar(c));
                            }
                            ++this.i;
                            Object value = this.parse();
                            dict.putEntry((String)key, value);
                            c = this.next();
                            if (c != ',') {
                                if (c == '}') break;
                                throw Starlark.errorf("in object, got %s, want ',' or '}'", Json.quoteChar(c));
                            }
                            ++this.i;
                        }
                    }
                    ++this.i;
                    return dict;
                }
                default: {
                    if (!Json.isdigit(c) && c != '-') break;
                    return this.parseNumber(c);
                }
            }
            throw Starlark.errorf("unexpected character %s", Json.quoteChar(c));
        }

        private String parseString() throws EvalException {
            ++this.i;
            StringBuilder str = new StringBuilder();
            block9: while (this.i < this.s.length()) {
                char c = this.s.charAt(this.i);
                if (c == '\"') {
                    ++this.i;
                    return str.toString();
                }
                if (c != '\\') {
                    if (c <= '\u001f') {
                        throw Starlark.errorf("invalid character '\\x%02x' in string literal", c);
                    }
                    ++this.i;
                    str.append(c);
                    continue;
                }
                ++this.i;
                if (this.i == this.s.length()) {
                    throw Starlark.errorf("incomplete escape", new Object[0]);
                }
                c = this.s.charAt(this.i);
                ++this.i;
                switch (c) {
                    case '\"': 
                    case '/': 
                    case '\\': {
                        str.append(c);
                        continue block9;
                    }
                    case 'b': {
                        str.append('\b');
                        continue block9;
                    }
                    case 'f': {
                        str.append('\f');
                        continue block9;
                    }
                    case 'n': {
                        str.append('\n');
                        continue block9;
                    }
                    case 'r': {
                        str.append('\r');
                        continue block9;
                    }
                    case 't': {
                        str.append('\t');
                        continue block9;
                    }
                    case 'u': {
                        if (this.i + 4 >= this.s.length()) {
                            throw Starlark.errorf("incomplete \\uXXXX escape", new Object[0]);
                        }
                        int hex = 0;
                        for (int j = 0; j < 4; ++j) {
                            c = this.s.charAt(this.i + j);
                            int nybble = 0;
                            if (Json.isdigit(c)) {
                                nybble = c - 48;
                            } else if ('a' <= c && c <= 'f') {
                                nybble = 10 + c - 97;
                            } else if ('A' <= c && c <= 'F') {
                                nybble = 10 + c - 65;
                            } else {
                                throw Starlark.errorf("invalid hex char %s in \\uXXXX escape", Json.quoteChar(c));
                            }
                            hex = hex << 4 | nybble;
                        }
                        str.append((char)hex);
                        this.i += 4;
                        continue block9;
                    }
                }
                throw Starlark.errorf("invalid escape '\\%s'", Character.valueOf(c));
            }
            throw Starlark.errorf("unclosed string literal", new Object[0]);
        }

        private Object parseNumber(char c) throws EvalException {
            boolean isfloat = false;
            int j = this.i;
            for (j = this.i + 1; j < this.s.length(); ++j) {
                c = this.s.charAt(j);
                if (Json.isdigit(c)) continue;
                if (c != '.' && c != 'e' && c != 'E' && c != '+' && c != '-') break;
                isfloat = true;
            }
            String num = this.s.substring(this.i, j);
            int digits = this.i;
            if (this.s.charAt(this.i) == '-') {
                ++digits;
            }
            if (digits == j || this.s.charAt(digits) == '.' || this.s.charAt(j - 1) == '.' || num.contains(".e") || this.s.charAt(digits) == '0' && j - digits > 1 && Json.isdigit(this.s.charAt(digits + 1))) {
                throw Starlark.errorf("invalid number: %s", num);
            }
            this.i = j;
            try {
                if (isfloat) {
                    double x = Double.parseDouble(num);
                    return StarlarkFloat.of(x);
                }
                return StarlarkInt.parse(num, 10);
            }
            catch (NumberFormatException unused) {
                throw Starlark.errorf("invalid number: %s", num);
            }
        }

        private boolean skipSpace() {
            while (this.i < this.s.length()) {
                char c = this.s.charAt(this.i);
                if (c != ' ' && c != '\t' && c != '\n' && c != '\r') {
                    return true;
                }
                ++this.i;
            }
            return false;
        }

        private char next() throws EvalException {
            if (this.skipSpace()) {
                return this.s.charAt(this.i);
            }
            throw Starlark.errorf("unexpected end of file", new Object[0]);
        }
    }

    private static final class Encoder {
        private final StringBuilder out = new StringBuilder();

        private Encoder() {
        }

        private void encode(Object x) throws EvalException {
            if (x == Starlark.NONE) {
                this.out.append("null");
                return;
            }
            if (x instanceof String) {
                this.appendQuoted((String)x);
                return;
            }
            if (x instanceof Boolean || x instanceof StarlarkInt) {
                this.out.append(x);
                return;
            }
            if (x instanceof StarlarkFloat) {
                if (!Double.isFinite(((StarlarkFloat)x).toDouble())) {
                    throw Starlark.errorf("cannot encode non-finite float %s", x);
                }
                this.out.append(x.toString());
                return;
            }
            if (x instanceof Encodable) {
                this.out.append(((Encodable)x).encodeJSON());
                return;
            }
            if (x instanceof Map) {
                Object[] keys;
                Map m4 = (Map)x;
                for (Object key : keys = m4.keySet().toArray()) {
                    if (key instanceof String) continue;
                    throw Starlark.errorf("%s has %s key, want string", Starlark.type(x), Starlark.type(key));
                }
                Arrays.sort(keys);
                this.out.append('{');
                String sep = "";
                for (Object key : keys) {
                    this.out.append(sep);
                    sep = ",";
                    this.appendQuoted((String)key);
                    this.out.append(':');
                    try {
                        this.encode(m4.get(key));
                    }
                    catch (EvalException ex) {
                        throw Starlark.errorf("in %s key %s: %s", Starlark.type(x), Starlark.repr(key), ex.getMessage());
                    }
                }
                this.out.append('}');
                return;
            }
            if (x instanceof StarlarkIterable) {
                this.out.append('[');
                String sep = "";
                int i = 0;
                for (Object elem : (StarlarkIterable)x) {
                    this.out.append(sep);
                    sep = ",";
                    try {
                        this.encode(elem);
                    }
                    catch (EvalException ex) {
                        throw Starlark.errorf("at %s index %d: %s", Starlark.type(x), i, ex.getMessage());
                    }
                    ++i;
                }
                this.out.append(']');
                return;
            }
            if (x instanceof Structure) {
                Structure obj = (Structure)x;
                Object[] fields = obj.getFieldNames().toArray(new String[0]);
                Arrays.sort(fields);
                this.out.append('{');
                String sep = "";
                for (Object field : fields) {
                    this.out.append(sep);
                    sep = ",";
                    this.appendQuoted((String)field);
                    this.out.append(":");
                    try {
                        Object v = obj.getValue((String)field);
                        this.encode(v);
                    }
                    catch (EvalException ex) {
                        throw Starlark.errorf("in %s field .%s: %s", Starlark.type(x), field, ex.getMessage());
                    }
                }
                this.out.append('}');
                return;
            }
            throw Starlark.errorf("cannot encode %s as JSON", Starlark.type(x));
        }

        private void appendQuoted(String s2) {
            this.out.append('\"');
            int i = 0;
            int n = s2.length();
            while (i < n) {
                int cp = s2.codePointAt(i);
                if (cp < 32) {
                    switch (cp) {
                        case 8: {
                            this.out.append("\\b");
                            break;
                        }
                        case 12: {
                            this.out.append("\\f");
                            break;
                        }
                        case 10: {
                            this.out.append("\\n");
                            break;
                        }
                        case 13: {
                            this.out.append("\\r");
                            break;
                        }
                        case 9: {
                            this.out.append("\\t");
                            break;
                        }
                        default: {
                            this.out.append("\\u00");
                            this.out.append(HEX[cp >> 4 & 0xF]);
                            this.out.append(HEX[cp & 0xF]);
                        }
                    }
                    ++i;
                    continue;
                }
                if (cp < 128) {
                    if (cp == 34 || cp == 92) {
                        this.out.append('\\');
                    }
                    this.out.append((char)cp);
                    ++i;
                    continue;
                }
                if (55296 <= cp && cp <= 57343) {
                    cp = 65533;
                }
                this.out.appendCodePoint(cp);
                i += Character.charCount(cp);
            }
            this.out.append('\"');
        }
    }

    public static interface Encodable {
        public String encodeJSON();
    }
}

