/*
 * Decompiled with CFR 0.152.
 */
package net.starlark.java.eval;

import com.google.common.collect.ImmutableList;
import com.google.common.flogger.GoogleLogger;
import java.io.ByteArrayOutputStream;
import java.io.FileDescriptor;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.zip.GZIPOutputStream;
import javax.annotation.Nullable;
import net.starlark.java.eval.Debug;
import net.starlark.java.eval.JNI;
import net.starlark.java.eval.StarlarkCallable;
import net.starlark.java.eval.StarlarkFunction;
import net.starlark.java.eval.StarlarkThread;
import net.starlark.java.syntax.Location;

final class CpuProfiler {
    private static final GoogleLogger logger;
    private final PprofWriter pprof;
    @Nullable
    private static volatile CpuProfiler instance;
    private static final Map<Integer, StarlarkThread> threads;
    private static FileInputStream pipe;

    private CpuProfiler(OutputStream out, Duration period) {
        this.pprof = new PprofWriter(out, period);
    }

    @Nullable
    static CpuProfiler get() {
        return instance;
    }

    @Nullable
    static StarlarkThread setStarlarkThread(StarlarkThread thread) {
        if (thread == null) {
            return threads.remove(CpuProfiler.gettid());
        }
        return threads.put(CpuProfiler.gettid(), thread);
    }

    static boolean start(OutputStream out, Duration period) {
        if (!CpuProfiler.supported()) {
            ((GoogleLogger.Api)logger.atWarning()).log("--starlark_cpu_profile is unsupported on this platform");
            return false;
        }
        if (instance != null) {
            throw new IllegalStateException("profiler started twice without intervening stop");
        }
        CpuProfiler.startRouter();
        if (!CpuProfiler.startTimer(period.toNanos() / 1000L)) {
            throw new IllegalStateException("profile signal handler already in use");
        }
        instance = new CpuProfiler(out, period);
        return true;
    }

    static void stop() throws IOException {
        if (instance == null) {
            throw new IllegalStateException("stop without start");
        }
        CpuProfiler profiler = instance;
        instance = null;
        CpuProfiler.stopTimer();
        profiler.pprof.writeEnd();
    }

    void addEvent(int ticks, ImmutableList<Debug.Frame> stack) {
        this.pprof.writeEvent(ticks, stack);
    }

    private static synchronized void startRouter() {
        if (pipe == null) {
            pipe = new FileInputStream(CpuProfiler.createPipe());
            Thread router = new Thread(CpuProfiler::router, "SIGPROF router");
            router.setDaemon(true);
            router.start();
        }
    }

    private static void router() {
        byte[] buf = new byte[4];
        while (true) {
            try {
                int n = pipe.read(buf);
                if (n < 0) {
                    throw new IllegalStateException("pipe closed");
                }
                if (n != 4) {
                    throw new IllegalStateException("short read");
                }
            }
            catch (IOException ex) {
                throw new IllegalStateException("unexpected I/O error", ex);
            }
            int tid = CpuProfiler.int32be(buf);
            StarlarkThread thread = threads.get(tid);
            if (thread == null) continue;
            thread.cpuTicks.getAndIncrement();
        }
    }

    private static int int32be(byte[] b) {
        return b[0] << 24 | (b[1] & 0xFF) << 16 | (b[2] & 0xFF) << 8 | b[3] & 0xFF;
    }

    private static native boolean supported();

    private static native FileDescriptor createPipe();

    private static native boolean startTimer(long var0);

    private static native void stopTimer();

    private static native int gettid();

    static {
        JNI.load();
        logger = GoogleLogger.forEnclosingClass();
        threads = new ConcurrentHashMap<Integer, StarlarkThread>();
    }

    private static final class PprofWriter {
        private final Duration period;
        private final long startNano;
        private GZIPOutputStream gz;
        private IOException error;
        private static final int PROFILE_SAMPLE_TYPE = 1;
        private static final int PROFILE_SAMPLE = 2;
        private static final int PROFILE_MAPPING = 3;
        private static final int PROFILE_LOCATION = 4;
        private static final int PROFILE_FUNCTION = 5;
        private static final int PROFILE_STRING_TABLE = 6;
        private static final int PROFILE_TIME_NANOS = 9;
        private static final int PROFILE_DURATION_NANOS = 10;
        private static final int PROFILE_PERIOD_TYPE = 11;
        private static final int PROFILE_PERIOD = 12;
        private static final int VALUETYPE_TYPE = 1;
        private static final int VALUETYPE_UNIT = 2;
        private static final int SAMPLE_LOCATION_ID = 1;
        private static final int SAMPLE_VALUE = 2;
        private static final int SAMPLE_LABEL = 3;
        private static final int LABEL_KEY = 1;
        private static final int LABEL_STR = 2;
        private static final int LABEL_NUM = 3;
        private static final int LABEL_NUM_UNIT = 4;
        private static final int LOCATION_ID = 1;
        private static final int LOCATION_MAPPING_ID = 2;
        private static final int LOCATION_ADDRESS = 3;
        private static final int LOCATION_LINE = 4;
        private static final int LINE_FUNCTION_ID = 1;
        private static final int LINE_LINE = 2;
        private static final int FUNCTION_ID = 1;
        private static final int FUNCTION_NAME = 2;
        private static final int FUNCTION_SYSTEM_NAME = 3;
        private static final int FUNCTION_FILENAME = 4;
        private static final int FUNCTION_START_LINE = 5;
        private final Map<String, Long> stringIDs = new HashMap<String, Long>();
        private final Map<Long, Long> functionIDs = new HashMap<Long, Long>();
        private final Map<Long, Long> locationIDs = new HashMap<Long, Long>();

        PprofWriter(OutputStream out, Duration period) {
            this.period = period;
            this.startNano = System.nanoTime();
            try {
                this.gz = new GZIPOutputStream(out);
                this.getStringID("");
                ByteArrayOutputStream unit = new ByteArrayOutputStream();
                PprofWriter.writeLong(unit, 1, this.getStringID("CPU"));
                PprofWriter.writeLong(unit, 2, this.getStringID("microseconds"));
                PprofWriter.writeByteArray(this.gz, 1, unit.toByteArray());
                PprofWriter.writeLong(this.gz, 12, period.toNanos() / 1000L);
                PprofWriter.writeByteArray(this.gz, 11, unit.toByteArray());
                PprofWriter.writeLong(this.gz, 9, System.currentTimeMillis() * 1000000L);
            }
            catch (IOException ex) {
                this.error = ex;
            }
        }

        synchronized void writeEvent(int ticks, ImmutableList<Debug.Frame> stack) {
            if (this.error == null) {
                try {
                    ByteArrayOutputStream sample = new ByteArrayOutputStream();
                    PprofWriter.writeLong(sample, 2, (long)ticks * this.period.toNanos() / 1000L);
                    for (Debug.Frame fr : stack.reverse()) {
                        PprofWriter.writeLong(sample, 1, this.getLocationID(fr));
                    }
                    PprofWriter.writeByteArray(this.gz, 2, sample.toByteArray());
                }
                catch (IOException ex) {
                    this.error = ex;
                }
            }
        }

        synchronized void writeEnd() throws IOException {
            long endNano = System.nanoTime();
            try {
                PprofWriter.writeLong(this.gz, 10, endNano - this.startNano);
                if (this.error != null) {
                    throw this.error;
                }
            }
            finally {
                this.gz.close();
            }
        }

        private static void writeLong(OutputStream out, int fieldNumber, long x) throws IOException {
            PprofWriter.writeVarint(out, fieldNumber << 3 | 0);
            PprofWriter.writeVarint(out, x);
        }

        private static void writeString(OutputStream out, int fieldNumber, String x) throws IOException {
            PprofWriter.writeByteArray(out, fieldNumber, x.getBytes(StandardCharsets.UTF_8));
        }

        private static void writeByteArray(OutputStream out, int fieldNumber, byte[] x) throws IOException {
            PprofWriter.writeVarint(out, fieldNumber << 3 | 2);
            PprofWriter.writeVarint(out, x.length);
            out.write(x);
        }

        private static void writeVarint(OutputStream out, long value) throws IOException {
            while ((value & 0xFFFFFFFFFFFFFF80L) != 0L) {
                out.write((byte)(value & 0x7FL | 0x80L));
                value >>>= 7;
            }
            out.write((byte)value);
        }

        private long getStringID(String s2) throws IOException {
            Long i = this.stringIDs.putIfAbsent(s2, Long.valueOf(this.stringIDs.size()));
            if (i == null) {
                PprofWriter.writeString(this.gz, 6, s2);
                return (long)this.stringIDs.size() - 1L;
            }
            return i;
        }

        private long getFunctionID(StarlarkCallable fn, long addr) throws IOException {
            Long id = this.functionIDs.get(addr);
            if (id == null) {
                id = addr;
                Location loc = fn.getLocation();
                String filename = loc.file();
                String name = fn.getName();
                if (name.equals("<toplevel>")) {
                    name = filename;
                }
                long nameID = this.getStringID(name);
                ByteArrayOutputStream fun = new ByteArrayOutputStream();
                PprofWriter.writeLong(fun, 1, id);
                PprofWriter.writeLong(fun, 2, nameID);
                PprofWriter.writeLong(fun, 3, nameID);
                PprofWriter.writeLong(fun, 4, this.getStringID(filename));
                PprofWriter.writeLong(fun, 5, loc.line());
                PprofWriter.writeByteArray(this.gz, 5, fun.toByteArray());
                this.functionIDs.put(addr, id);
            }
            return id;
        }

        private long getLocationID(Debug.Frame fr) throws IOException {
            Long id;
            StarlarkCallable fn = fr.getFunction();
            int fnAddr = System.identityHashCode(fn);
            long pcAddr = fnAddr;
            if (fn instanceof StarlarkFunction) {
                // empty if block
            }
            if ((id = this.locationIDs.get(pcAddr)) == null) {
                id = pcAddr;
                ByteArrayOutputStream line = new ByteArrayOutputStream();
                PprofWriter.writeLong(line, 1, this.getFunctionID(fn, fnAddr));
                PprofWriter.writeLong(line, 2, fr.getLocation().line());
                ByteArrayOutputStream loc = new ByteArrayOutputStream();
                PprofWriter.writeLong(loc, 1, id);
                PprofWriter.writeLong(loc, 3, pcAddr);
                PprofWriter.writeByteArray(loc, 4, line.toByteArray());
                PprofWriter.writeByteArray(this.gz, 4, loc.toByteArray());
                this.locationIDs.put(pcAddr, id);
            }
            return id;
        }
    }
}

