/*
 * Decompiled with CFR 0.152.
 */
package com.google.devtools.starlark.common;

import com.google.common.base.Preconditions;
import com.google.common.base.Splitter;
import com.google.common.collect.ImmutableList;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public final class DocstringUtils {
    private DocstringUtils() {
    }

    public static DocstringInfo parseDocstring(String doc, List<DocstringParseError> parseErrors) {
        DocstringParser parser = new DocstringParser(doc);
        DocstringInfo result = parser.parse();
        parseErrors.addAll(parser.errors);
        return result;
    }

    public static class DocstringParseError {
        final String message;
        final int lineNumber;
        final String line;

        public DocstringParseError(String message, int lineNumber, String line) {
            this.message = message;
            this.lineNumber = lineNumber;
            this.line = line;
        }

        public String toString() {
            return this.lineNumber + ": " + this.message;
        }

        public String getMessage() {
            return this.message;
        }

        public int getLineNumber() {
            return this.lineNumber;
        }

        public String getLine() {
            return this.line;
        }
    }

    private static class DocstringParser {
        private final String docstring;
        private int startOfLineOffset = 0;
        private int endOfLineOffset = 0;
        private int lineNumber = 0;
        private int baselineIndentation = 0;
        private boolean blankLineBefore = false;
        private boolean specialSectionsStarted = false;
        private ArrayList<String> originalLines = new ArrayList();
        private String line = "";
        private final List<DocstringParseError> errors = new ArrayList<DocstringParseError>();
        private static final Pattern paramLineMatcher = Pattern.compile("\\s*(?<name>[*\\w]+)( \\(\\s*(?<attributes>.*)\\s*\\))?: (?<description>.*)");
        private static final Pattern attributesSeparator = Pattern.compile("\\s*,\\s*");

        private DocstringParser(String docstring) {
            this.docstring = docstring;
            int indentation = Integer.MAX_VALUE;
            boolean first = true;
            for (String line : Splitter.on("\n").split(docstring)) {
                int i;
                if (first) {
                    first = false;
                    continue;
                }
                for (i = 0; i < line.length() && line.charAt(i) == ' '; ++i) {
                }
                if (i == line.length()) continue;
                indentation = Math.min(indentation, i);
            }
            if (indentation == Integer.MAX_VALUE) {
                indentation = 0;
            }
            this.nextLine();
            this.baselineIndentation = indentation;
        }

        private boolean nextLine() {
            if (this.startOfLineOffset >= this.docstring.length()) {
                return false;
            }
            this.blankLineBefore = this.line.trim().isEmpty();
            this.startOfLineOffset = this.endOfLineOffset;
            if (this.startOfLineOffset >= this.docstring.length()) {
                this.line = "";
                return false;
            }
            if (this.docstring.charAt(this.startOfLineOffset) == '\n') {
                ++this.startOfLineOffset;
            }
            ++this.lineNumber;
            this.endOfLineOffset = this.docstring.indexOf(10, this.startOfLineOffset);
            if (this.endOfLineOffset < 0) {
                this.endOfLineOffset = this.docstring.length();
            }
            String originalLine = this.docstring.substring(this.startOfLineOffset, this.endOfLineOffset);
            this.originalLines.add(originalLine);
            int indentation = DocstringParser.getIndentation(originalLine);
            if (this.endOfLineOffset == this.docstring.length() && this.startOfLineOffset != 0) {
                if (!originalLine.trim().isEmpty()) {
                    this.error("closing docstring quote should be on its own line, indented the same as the opening quote");
                } else if (indentation != this.baselineIndentation) {
                    this.error("closing docstring quote should be indented the same as the opening quote");
                }
            }
            if (originalLine.trim().isEmpty()) {
                this.line = "";
            } else {
                if (indentation < this.baselineIndentation) {
                    this.error("line indented too little (here: " + indentation + " spaces; expected: " + this.baselineIndentation + " spaces)");
                    this.startOfLineOffset += indentation;
                } else {
                    this.startOfLineOffset += this.baselineIndentation;
                }
                this.line = this.docstring.substring(this.startOfLineOffset, this.endOfLineOffset);
            }
            return true;
        }

        private boolean onLastLine() {
            return this.endOfLineOffset >= this.docstring.length();
        }

        private boolean eof() {
            return this.startOfLineOffset >= this.docstring.length();
        }

        private static int getIndentation(String line) {
            int index;
            for (index = 0; index < line.length() && line.charAt(index) == ' '; ++index) {
            }
            return index;
        }

        private void error(String message) {
            this.error(this.lineNumber, message);
        }

        private void error(int lineNumber, String message) {
            this.errors.add(new DocstringParseError(message, lineNumber, this.originalLines.get(lineNumber - 1)));
        }

        private void parseArgumentSection(List<ParameterDoc> params, String returns, String deprecated) {
            this.checkSectionStart(!params.isEmpty());
            if (!returns.isEmpty()) {
                this.error("'Args:' section should go before the 'Returns:' section");
            }
            if (!deprecated.isEmpty()) {
                this.error("'Args:' section should go before the 'Deprecated:' section");
            }
            params.addAll(this.parseParameters());
        }

        DocstringInfo parse() {
            String summary = this.line;
            String nonStandardDeprecation = this.checkForNonStandardDeprecation(this.line);
            if (!this.nextLine()) {
                return new DocstringInfo(summary, Collections.emptyList(), "", nonStandardDeprecation, "");
            }
            if (!this.line.isEmpty()) {
                this.error("the one-line summary should be followed by a blank line");
            } else {
                this.nextLine();
            }
            ArrayList<String> longDescriptionLines = new ArrayList<String>();
            ArrayList<ParameterDoc> params = new ArrayList<ParameterDoc>();
            String returns = "";
            String deprecated = "";
            boolean descriptionBodyAfterSpecialSectionsReported = false;
            block12: while (!this.eof()) {
                switch (this.line) {
                    case "Args:": {
                        this.parseArgumentSection(params, returns, deprecated);
                        continue block12;
                    }
                    case "Arguments:": {
                        this.parseArgumentSection(params, returns, deprecated);
                        continue block12;
                    }
                    case "Returns:": {
                        this.checkSectionStart(!returns.isEmpty());
                        if (!deprecated.isEmpty()) {
                            this.error("'Returns:' section should go before the 'Deprecated:' section");
                        }
                        returns = this.parseSectionAfterHeading();
                        continue block12;
                    }
                    case "Deprecated:": {
                        this.checkSectionStart(!deprecated.isEmpty());
                        deprecated = this.parseSectionAfterHeading();
                        continue block12;
                    }
                }
                if (this.specialSectionsStarted && !descriptionBodyAfterSpecialSectionsReported) {
                    this.error("description body should go before the special sections");
                    descriptionBodyAfterSpecialSectionsReported = true;
                }
                if (deprecated.isEmpty() && nonStandardDeprecation.isEmpty()) {
                    nonStandardDeprecation = this.checkForNonStandardDeprecation(this.line);
                }
                if (this.line.startsWith("Returns: ")) {
                    this.error("the return value should be documented in a section, like this:\n\nReturns:\n  <documentation here>\n\nFor more details, please have a look at the documentation.");
                }
                if (!this.onLastLine() || !this.line.trim().isEmpty()) {
                    longDescriptionLines.add(this.line);
                }
                this.nextLine();
            }
            if (deprecated.isEmpty()) {
                deprecated = nonStandardDeprecation;
            }
            return new DocstringInfo(summary, params, returns, deprecated, String.join((CharSequence)"\n", longDescriptionLines));
        }

        private void checkSectionStart(boolean duplicateSection) {
            this.specialSectionsStarted = true;
            if (!this.blankLineBefore) {
                this.error("section should be preceded by a blank line");
            }
            if (duplicateSection) {
                this.error("duplicate '" + this.line + "' section");
            }
        }

        private String checkForNonStandardDeprecation(String line) {
            if (line.toLowerCase().startsWith("deprecated:") || line.contains("DEPRECATED")) {
                this.error("use a 'Deprecated:' section for deprecations, similar to a 'Returns:' section:\n\nDeprecated:\n  <reason and alternative>\n\nFor more details, please have a look at the documentation.");
                return line;
            }
            return "";
        }

        private List<ParameterDoc> parseParameters() {
            int sectionLineNumber = this.lineNumber;
            this.nextLine();
            ArrayList<ParameterDoc> params = new ArrayList<ParameterDoc>();
            int expectedParamLineIndentation = -1;
            while (!this.eof()) {
                if (this.line.isEmpty()) {
                    this.nextLine();
                    continue;
                }
                int actualIndentation = DocstringParser.getIndentation(this.line);
                if (actualIndentation == 0) {
                    if (this.blankLineBefore) break;
                    this.error("end of 'Args' section without blank line");
                    break;
                }
                if (expectedParamLineIndentation == -1) {
                    expectedParamLineIndentation = actualIndentation;
                }
                if (expectedParamLineIndentation != actualIndentation) {
                    this.error("inconsistent indentation of parameter lines (before: " + expectedParamLineIndentation + "; here: " + actualIndentation + " spaces)");
                }
                int paramLineNumber = this.lineNumber;
                String trimmedLine = this.line.substring(actualIndentation);
                Matcher matcher = paramLineMatcher.matcher(trimmedLine);
                if (!matcher.matches()) {
                    this.error("invalid parameter documentation (expected format: \"parameter_name: documentation\"). For more details, please have a look at the documentation.");
                    this.nextLine();
                    continue;
                }
                String parameterName = Preconditions.checkNotNull(matcher.group("name"));
                String attributesString = matcher.group("attributes");
                StringBuilder description = new StringBuilder(matcher.group("description"));
                List<String> attributes = attributesString == null ? Collections.emptyList() : Arrays.asList(attributesSeparator.split(attributesString));
                this.parseContinuedParamDescription(actualIndentation, description);
                String parameterDescription = description.toString().trim();
                if (parameterDescription.isEmpty()) {
                    this.error(paramLineNumber, "empty parameter description for '" + parameterName + "'");
                }
                params.add(new ParameterDoc(parameterName, attributes, parameterDescription));
            }
            if (params.isEmpty()) {
                this.error(sectionLineNumber, "section is empty or badly formatted");
            }
            return params;
        }

        private void parseContinuedParamDescription(int baselineIndentation, StringBuilder description) {
            ArrayList<String> buffer = new ArrayList<String>();
            int continuationIndentation = Integer.MAX_VALUE;
            while (this.nextLine()) {
                if (!this.line.isEmpty()) {
                    if (DocstringParser.getIndentation(this.line) <= baselineIndentation) break;
                    continuationIndentation = Math.min(DocstringParser.getIndentation(this.line), continuationIndentation);
                }
                buffer.add(this.line);
            }
            for (String bufLine : buffer) {
                description.append('\n');
                if (bufLine.isEmpty()) continue;
                String trimmedLine = bufLine.substring(continuationIndentation);
                description.append(trimmedLine);
            }
        }

        private String parseSectionAfterHeading() {
            String result;
            int sectionLineNumber = this.lineNumber;
            this.nextLine();
            StringBuilder contents = new StringBuilder();
            boolean firstLine = true;
            while (!this.eof()) {
                String trimmedLine;
                if (this.line.isEmpty()) {
                    trimmedLine = this.line;
                } else {
                    if (DocstringParser.getIndentation(this.line) == 0) {
                        if (this.blankLineBefore) break;
                        this.error("end of section without blank line");
                        break;
                    }
                    if (DocstringParser.getIndentation(this.line) < 2) {
                        this.error("text in a section has to be indented by two spaces (relative to the left margin of the docstring)");
                        trimmedLine = this.line.substring(DocstringParser.getIndentation(this.line));
                    } else {
                        trimmedLine = this.line.substring(2);
                    }
                }
                if (!firstLine) {
                    contents.append('\n');
                }
                contents.append(trimmedLine);
                this.nextLine();
                firstLine = false;
            }
            if ((result = contents.toString().trim()).isEmpty()) {
                this.error(sectionLineNumber, "section is empty");
            }
            return result;
        }
    }

    public static class ParameterDoc {
        final String parameterName;
        final List<String> attributes;
        final String description;

        public ParameterDoc(String parameterName, List<String> attributes, String description) {
            this.parameterName = parameterName;
            this.attributes = ImmutableList.copyOf(attributes);
            this.description = description;
        }

        public String getParameterName() {
            return this.parameterName;
        }

        public List<String> getAttributes() {
            return this.attributes;
        }

        public String getDescription() {
            return this.description;
        }
    }

    public static class DocstringInfo {
        final String summary;
        final List<ParameterDoc> parameters;
        final String returns;
        final String deprecated;
        final String longDescription;

        public DocstringInfo(String summary, List<ParameterDoc> parameters, String returns, String deprecated, String longDescription) {
            this.summary = summary;
            this.parameters = ImmutableList.copyOf(parameters);
            this.returns = returns;
            this.deprecated = deprecated;
            this.longDescription = longDescription;
        }

        public String getSummary() {
            return this.summary;
        }

        public List<ParameterDoc> getParameters() {
            return this.parameters;
        }

        public String getLongDescription() {
            return this.longDescription;
        }

        public String getDeprecated() {
            return this.deprecated;
        }

        public boolean isSingleLineDocstring() {
            return this.longDescription.isEmpty() && this.parameters.isEmpty() && this.returns.isEmpty();
        }

        public String getReturns() {
            return this.returns;
        }
    }
}

