/*
 * Decompiled with CFR 0.152.
 */
package com.google.devtools.build.lib.vfs;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.io.ByteSink;
import com.google.common.io.ByteSource;
import com.google.common.io.ByteStreams;
import com.google.devtools.build.lib.vfs.FileAccessException;
import com.google.devtools.build.lib.vfs.FileStatus;
import com.google.devtools.build.lib.vfs.FileSystem;
import com.google.devtools.build.lib.vfs.Path;
import com.google.devtools.build.lib.vfs.PathFragment;
import com.google.devtools.build.lib.vfs.Symlinks;
import com.google.errorprone.annotations.InlineMe;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.function.Predicate;
import javax.annotation.Nullable;

public class FileSystemUtils {
    private FileSystemUtils() {
    }

    public static void checkBaseName(String baseName) {
        if (baseName.length() == 0) {
            throw new IllegalArgumentException("Child must not be empty string ('')");
        }
        if (baseName.equals(".") || baseName.equals("..")) {
            throw new IllegalArgumentException("baseName must not be '" + baseName + "'");
        }
        if (baseName.indexOf(47) != -1) {
            throw new IllegalArgumentException("baseName must not contain a slash: '" + baseName + "'");
        }
    }

    public static Path commonAncestor(Path a, Path b) {
        while (a != null && !b.startsWith(a)) {
            a = a.getParentDirectory();
        }
        return a;
    }

    public static PathFragment commonAncestor(PathFragment a, PathFragment b) {
        while (a != null && !b.startsWith(a)) {
            a = a.getParentDirectory();
        }
        return a;
    }

    public static PathFragment relativePath(PathFragment fromDir, PathFragment to) {
        if (to.equals(fromDir)) {
            return PathFragment.EMPTY_FRAGMENT;
        }
        if (to.startsWith(fromDir)) {
            return to.relativeTo(fromDir);
        }
        PathFragment ancestor = FileSystemUtils.commonAncestor(fromDir, to);
        if (ancestor == null) {
            return to;
        }
        int levels = fromDir.relativeTo(ancestor).segmentCount();
        StringBuilder dotdots = new StringBuilder();
        for (int i = 0; i < levels; ++i) {
            dotdots.append("../");
        }
        return PathFragment.create(dotdots.toString()).getRelative(to.relativeTo(ancestor));
    }

    public static String removeExtension(String filename) {
        int lastDotIndex = filename.lastIndexOf(46);
        if (lastDotIndex == -1) {
            return filename;
        }
        int lastSlashIndex = filename.lastIndexOf(47);
        if (lastSlashIndex > lastDotIndex) {
            return filename;
        }
        return filename.substring(0, lastDotIndex);
    }

    public static PathFragment removeExtension(PathFragment path) {
        return path.replaceName(FileSystemUtils.removeExtension(path.getBaseName()));
    }

    public static Path removeExtension(Path path) {
        return path.getFileSystem().getPath(FileSystemUtils.removeExtension(path.asFragment()));
    }

    public static PathFragment replaceExtension(PathFragment path, String newExtension) {
        return path.replaceName(FileSystemUtils.removeExtension(path.getBaseName()) + newExtension);
    }

    @Nullable
    public static PathFragment replaceExtension(PathFragment path, String newExtension, String oldExtension) {
        String base = path.getBaseName();
        if (!base.endsWith(oldExtension)) {
            return null;
        }
        String newBase = base.substring(0, base.length() - oldExtension.length()) + newExtension;
        return path.replaceName(newBase);
    }

    @Nullable
    public static Path replaceExtension(Path path, String newExtension) {
        PathFragment fragment = FileSystemUtils.replaceExtension(path.asFragment(), newExtension);
        return fragment == null ? null : path.getFileSystem().getPath(fragment);
    }

    public static PathFragment appendExtension(PathFragment path, String newExtension) {
        return path.replaceName(path.getBaseName() + newExtension);
    }

    public static PathFragment appendWithoutExtension(PathFragment path, String toAppend) {
        return path.replaceName(FileSystemUtils.appendWithoutExtension(path.getBaseName(), toAppend));
    }

    private static String appendWithoutExtension(String name, String toAppend) {
        int dotIndex = name.lastIndexOf(46);
        if (dotIndex > 0) {
            String baseName = name.substring(0, dotIndex);
            String extension = name.substring(dotIndex);
            return baseName + toAppend + extension;
        }
        return name + toAppend;
    }

    public static Path getWorkingDirectory(FileSystem fs) {
        return fs.getPath(FileSystemUtils.getWorkingDirectory());
    }

    public static PathFragment getWorkingDirectory() {
        return PathFragment.create(System.getProperty("user.dir", "/"));
    }

    public static void touchFile(Path path) throws IOException {
        if (path.exists()) {
            path.setLastModifiedTime(-1L);
        } else {
            FileSystemUtils.createEmptyFile(path);
        }
    }

    public static void createEmptyFile(Path path) throws IOException {
        path.getOutputStream().close();
    }

    public static void ensureSymbolicLink(Path link, Path target) throws IOException {
        FileSystemUtils.ensureSymbolicLink(link, target.asFragment());
    }

    public static void ensureSymbolicLink(Path link, String target) throws IOException {
        FileSystemUtils.ensureSymbolicLink(link, PathFragment.create(target));
    }

    public static void ensureSymbolicLink(Path link, PathFragment target) throws IOException {
        block7: {
            try {
                if (link.readSymbolicLink().equals(target)) {
                    return;
                }
            }
            catch (IOException iOException) {
                // empty catch block
            }
            if (link.isSymbolicLink()) {
                link.delete();
            } else {
                link.getParentDirectory().createDirectoryAndParents();
            }
            try {
                link.createSymbolicLink(target);
            }
            catch (IOException e) {
                if (link.isSymbolicLink() && link.resolveSymbolicLinks().equals(link.getRelative(target))) break block7;
                throw e;
            }
        }
    }

    public static ByteSource asByteSource(final Path path) {
        return new ByteSource(){

            @Override
            public InputStream openStream() throws IOException {
                return path.getInputStream();
            }
        };
    }

    public static ByteSink asByteSink(final Path path, final boolean append) {
        return new ByteSink(){

            @Override
            public OutputStream openStream() throws IOException {
                return path.getOutputStream(append);
            }
        };
    }

    public static ByteSink asByteSink(Path path) {
        return FileSystemUtils.asByteSink(path, false);
    }

    public static void copyFile(Path from, Path to) throws IOException {
        try {
            to.delete();
        }
        catch (IOException e) {
            throw new IOException("error copying file: couldn't delete destination: " + e.getMessage());
        }
        try (InputStream in = from.getInputStream();
             OutputStream out = to.getOutputStream();){
            ByteStreams.copy(in, out);
        }
        to.setLastModifiedTime(from.getLastModifiedTime());
        if (!from.isWritable()) {
            to.setWritable(false);
        }
        to.setExecutable(from.isExecutable());
    }

    private static long copyLargeBuffer(InputStream from, OutputStream to) throws IOException {
        int r;
        byte[] buf = new byte[0x100000];
        long total = 0L;
        while ((r = from.read(buf)) != -1) {
            to.write(buf, 0, r);
            total += (long)r;
        }
        return total;
    }

    public static MoveResult moveFile(Path from, Path to) throws IOException {
        to.delete();
        try {
            from.renameTo(to);
            return MoveResult.FILE_MOVED;
        }
        catch (IOException unused) {
            FileStatus stat = from.stat(Symlinks.NOFOLLOW);
            if (stat.isFile()) {
                block36: {
                    try (InputStream in = from.getInputStream();
                         OutputStream out = to.getOutputStream();){
                        FileSystemUtils.copyLargeBuffer(in, out);
                    }
                    catch (FileAccessException e) {
                        if (!from.isReadable()) {
                            from.setReadable(true);
                            try (InputStream in2 = from.getInputStream();
                                 OutputStream out2 = to.getOutputStream();){
                                FileSystemUtils.copyLargeBuffer(in2, out2);
                                break block36;
                            }
                        }
                        throw e;
                    }
                }
                to.setLastModifiedTime(stat.getLastModifiedTime());
                if (!from.isWritable()) {
                    to.setWritable(false);
                }
                to.setExecutable(from.isExecutable());
            } else if (stat.isSymbolicLink()) {
                to.createSymbolicLink(from.readSymbolicLink());
            } else {
                throw new IOException("Don't know how to copy " + from);
            }
            if (!from.delete()) {
                if (!to.delete()) {
                    throw new IOException("Unable to delete " + to);
                }
                throw new IOException("Unable to delete " + from);
            }
            return MoveResult.FILE_COPIED;
        }
    }

    public static Path copyTool(Path source, Path target) throws IOException {
        FileStatus sourceStat = null;
        FileStatus targetStat = target.statNullable();
        if (targetStat != null) {
            sourceStat = source.stat(Symlinks.FOLLOW);
        }
        if (targetStat == null || targetStat.getLastModifiedTime() != sourceStat.getLastModifiedTime() || targetStat.getSize() != sourceStat.getSize()) {
            FileSystemUtils.copyFile(source, target);
            target.setWritable(source.isWritable());
            target.setExecutable(source.isExecutable());
            target.setLastModifiedTime(source.getLastModifiedTime());
        }
        return target;
    }

    public static Collection<Path> traverseTree(Path root, Predicate<Path> predicate) throws IOException {
        ArrayList<Path> paths = new ArrayList<Path>();
        FileSystemUtils.traverseTree(paths, root, predicate);
        return paths;
    }

    public static void traverseTree(Collection<Path> paths, Path root, Predicate<Path> predicate) throws IOException {
        for (Path p : root.getDirectoryEntries()) {
            if (predicate.test(p)) {
                paths.add(p);
            }
            if (!p.isDirectory(Symlinks.NOFOLLOW)) continue;
            FileSystemUtils.traverseTree(paths, p, predicate);
        }
    }

    public static void copyTreesBelow(Path from, Path to, Symlinks followSymlinks) throws IOException {
        if (to.startsWith(from)) {
            throw new IllegalArgumentException(to + " is a subdirectory of " + from);
        }
        Collection<Path> entries = from.getDirectoryEntries();
        for (Path entry : entries) {
            Path toPath = to.getChild(entry.getBaseName());
            if (!followSymlinks.toBoolean() && entry.isSymbolicLink()) {
                FileSystemUtils.ensureSymbolicLink(toPath, entry.readSymbolicLink());
                continue;
            }
            if (entry.isFile()) {
                FileSystemUtils.copyFile(entry, toPath);
                continue;
            }
            toPath.createDirectory();
            FileSystemUtils.copyTreesBelow(entry, toPath, followSymlinks);
        }
    }

    public static void moveTreesBelow(Path from, Path to) throws IOException {
        if (to.startsWith(from)) {
            throw new IllegalArgumentException(to + " is a subdirectory of " + from);
        }
        Collection<Path> entries = from.getDirectoryEntries();
        for (Path entry : entries) {
            if (entry.isDirectory(Symlinks.NOFOLLOW)) {
                Path subDir = to.getChild(entry.getBaseName());
                subDir.createDirectory();
                FileSystemUtils.moveTreesBelow(entry, subDir);
                continue;
            }
            Path newEntry = to.getChild(entry.getBaseName());
            FileSystemUtils.moveFile(entry, newEntry);
        }
    }

    @Deprecated
    @InlineMe(replacement="dir.createDirectoryAndParents()")
    public static void createDirectoryAndParents(Path dir) throws IOException {
        dir.createDirectoryAndParents();
    }

    public static boolean removeDirectoryAndParents(Path base, PathFragment toRemove) {
        if (toRemove.isAbsolute()) {
            return false;
        }
        try {
            while (!toRemove.isEmpty()) {
                Path p = base.getRelative(toRemove);
                if (p.exists()) {
                    p.delete();
                }
                toRemove = toRemove.getParentDirectory();
            }
        }
        catch (IOException e) {
            return false;
        }
        return true;
    }

    public static char[] convertFromLatin1(byte[] content) {
        char[] latin1 = new char[content.length];
        for (int i = 0; i < latin1.length; ++i) {
            latin1[i] = (char)(0xFF & content[i]);
        }
        return latin1;
    }

    public static void writeIsoLatin1(Path file, String ... lines) throws IOException {
        FileSystemUtils.writeLinesAs(file, StandardCharsets.ISO_8859_1, lines);
    }

    public static void appendIsoLatin1(Path file, String ... lines) throws IOException {
        FileSystemUtils.appendLinesAs(file, StandardCharsets.ISO_8859_1, lines);
    }

    public static void writeContentAsLatin1(Path outputFile, String content) throws IOException {
        FileSystemUtils.writeContent(outputFile, StandardCharsets.ISO_8859_1, content);
    }

    public static void writeContent(Path outputFile, Charset charset, String content) throws IOException {
        FileSystemUtils.asByteSink(outputFile).asCharSink(charset).write(content);
    }

    public static void writeContent(Path outputFile, byte[] content) throws IOException {
        FileSystemUtils.asByteSink(outputFile).write(content);
    }

    public static void writeLinesAs(Path file, Charset charset, String ... lines) throws IOException {
        FileSystemUtils.writeLinesAs(file, charset, Arrays.asList(lines));
    }

    public static void writeLinesAs(Path file, Charset charset, Iterable<String> lines) throws IOException {
        file.getParentDirectory().createDirectoryAndParents();
        FileSystemUtils.asByteSink(file).asCharSink(charset).writeLines(lines);
    }

    public static void appendLinesAs(Path file, Charset charset, String ... lines) throws IOException {
        FileSystemUtils.appendLinesAs(file, charset, Arrays.asList(lines));
    }

    public static void appendLinesAs(Path file, Charset charset, Iterable<String> lines) throws IOException {
        file.getParentDirectory().createDirectoryAndParents();
        FileSystemUtils.asByteSink(file, true).asCharSink(charset).writeLines(lines);
    }

    public static void maybeUpdateContent(Path outputFile, byte[] newContent) throws IOException {
        byte[] currentContent;
        try {
            currentContent = FileSystemUtils.readContent(outputFile);
        }
        catch (IOException e) {
            currentContent = null;
        }
        if (currentContent == null) {
            FileSystemUtils.writeContent(outputFile, newContent);
        } else if (!Arrays.equals(newContent, currentContent)) {
            if (!outputFile.isWritable()) {
                outputFile.delete();
            }
            FileSystemUtils.writeContent(outputFile, newContent);
        }
    }

    public static char[] readContentAsLatin1(InputStream in) throws IOException {
        return FileSystemUtils.convertFromLatin1(ByteStreams.toByteArray(in));
    }

    public static char[] readContentAsLatin1(Path inputFile) throws IOException {
        return FileSystemUtils.convertFromLatin1(FileSystemUtils.readContent(inputFile));
    }

    public static ImmutableList<String> readLinesAsLatin1(Path inputFile) throws IOException {
        return FileSystemUtils.readLines(inputFile, StandardCharsets.ISO_8859_1);
    }

    public static ImmutableList<String> readLines(Path inputFile, Charset charset) throws IOException {
        return FileSystemUtils.asByteSource(inputFile).asCharSource(charset).readLines();
    }

    public static byte[] readContent(Path inputFile) throws IOException {
        return FileSystemUtils.asByteSource(inputFile).read();
    }

    public static String readContent(Path inputFile, Charset charset) throws IOException {
        return FileSystemUtils.asByteSource(inputFile).asCharSource(charset).read();
    }

    public static byte[] readContentWithLimit(Path inputFile, int limit) throws IOException {
        Preconditions.checkArgument(limit >= 0, "limit needs to be >=0, but it is %s", limit);
        ByteSource byteSource = FileSystemUtils.asByteSource(inputFile);
        byte[] buffer = new byte[limit];
        try (InputStream inputStream = byteSource.openBufferedStream();){
            int read = ByteStreams.read(inputStream, buffer, 0, limit);
            byte[] byArray = read == limit ? buffer : Arrays.copyOf(buffer, read);
            return byArray;
        }
    }

    public static byte[] readWithKnownFileSize(Path path, long fileSize) throws IOException {
        if (fileSize > Integer.MAX_VALUE) {
            throw new IOException("Cannot read file with size larger than 2GB");
        }
        int fileSizeInt = (int)fileSize;
        byte[] bytes = FileSystemUtils.readContentWithLimit(path, fileSizeInt);
        if (fileSizeInt > bytes.length) {
            throw new ShortReadIOException(path, fileSizeInt, bytes.length);
        }
        return bytes;
    }

    public static String getFileSystem(Path path) {
        return path.getFileSystem().getFileSystemType(path.asFragment());
    }

    public static boolean startsWithAny(Path path, Iterable<Path> prefixes) {
        for (Path prefix : prefixes) {
            if (!path.startsWith(prefix)) continue;
            return true;
        }
        return false;
    }

    public static boolean startsWithAny(PathFragment path, Iterable<PathFragment> prefixes) {
        for (PathFragment prefix : prefixes) {
            if (!path.startsWith(prefix)) continue;
            return true;
        }
        return false;
    }

    public static void createHardLink(Path linkPath, Path originalPath) throws IOException {
        if (originalPath.isDirectory()) {
            for (Path originalSubpath : originalPath.getDirectoryEntries()) {
                Path linkSubpath = linkPath.getRelative(originalSubpath.relativeTo(originalPath));
                FileSystemUtils.createHardLink(linkSubpath, originalSubpath);
            }
        } else {
            Path parentDir = linkPath.getParentDirectory();
            if (!parentDir.exists()) {
                parentDir.createDirectoryAndParents();
            }
            originalPath.createHardLink(linkPath);
        }
    }

    public static class ShortReadIOException
    extends IOException {
        public final Path path;
        public final int fileSize;
        public final int numBytesRead;

        private ShortReadIOException(Path path, int fileSize, int numBytesRead) {
            super("Unexpected short read from file '" + path + "' (expected " + fileSize + ", got " + numBytesRead + " bytes)");
            this.path = path;
            this.fileSize = fileSize;
            this.numBytesRead = numBytesRead;
        }
    }

    public static enum MoveResult {
        FILE_MOVED,
        FILE_COPIED;

    }
}

