/*
 * Decompiled with CFR 0.152.
 */
package com.google.devtools.build.lib.vfs;

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.io.ByteSource;
import com.google.common.io.CharStreams;
import com.google.devtools.build.lib.vfs.DigestHashFunction;
import com.google.devtools.build.lib.vfs.Dirent;
import com.google.devtools.build.lib.vfs.FileStatus;
import com.google.devtools.build.lib.vfs.FileSymlinkLoopException;
import com.google.devtools.build.lib.vfs.Path;
import com.google.devtools.build.lib.vfs.PathFragment;
import com.google.devtools.build.lib.vfs.Root;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.nio.channels.ReadableByteChannel;
import java.nio.channels.SeekableByteChannel;
import java.nio.charset.StandardCharsets;
import java.nio.file.FileAlreadyExistsException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import javax.annotation.Nullable;

public abstract class FileSystem {
    private final DigestHashFunction digestFunction;
    private final Root absoluteRoot = new Root.AbsoluteRoot(this);

    public FileSystem(DigestHashFunction digestFunction) {
        this.digestFunction = Preconditions.checkNotNull(digestFunction);
    }

    public DigestHashFunction getDigestFunction() {
        return this.digestFunction;
    }

    public Path getPath(String path) {
        return Path.create(path, this);
    }

    public Path getPath(PathFragment pathFragment) {
        return Path.create(pathFragment, this);
    }

    final Root getAbsoluteRoot() {
        return this.absoluteRoot;
    }

    public abstract boolean supportsModifications(PathFragment var1);

    public abstract boolean supportsSymbolicLinksNatively(PathFragment var1);

    protected abstract boolean supportsHardLinksNatively(PathFragment var1);

    public abstract boolean isFilePathCaseSensitive();

    public String getFileSystemType(PathFragment path) {
        String fileSystem = "unknown";
        int bestMountPointSegmentCount = -1;
        try {
            Path canonicalPath = this.resolveSymbolicLinks(path);
            PathFragment mountTable = PathFragment.createAlreadyNormalized("/proc/mounts");
            try (InputStreamReader reader = new InputStreamReader(this.getInputStream(mountTable), StandardCharsets.ISO_8859_1);){
                for (String line : CharStreams.readLines(reader)) {
                    String[] words = line.split("\\s+");
                    if (words.length < 3 || !words[1].startsWith("/")) continue;
                    PathFragment mountPoint = PathFragment.create(words[1]);
                    int segmentCount = mountPoint.segmentCount();
                    if (!canonicalPath.startsWith(mountPoint) || segmentCount <= bestMountPointSegmentCount) continue;
                    bestMountPointSegmentCount = segmentCount;
                    fileSystem = words[2];
                }
            }
        }
        catch (IOException iOException) {
            // empty catch block
        }
        return fileSystem;
    }

    public abstract boolean createDirectory(PathFragment var1) throws IOException;

    protected abstract boolean createWritableDirectory(PathFragment var1) throws IOException;

    public abstract void createDirectoryAndParents(PathFragment var1) throws IOException;

    protected abstract long getFileSize(PathFragment var1, boolean var2) throws IOException;

    protected abstract boolean delete(PathFragment var1) throws IOException;

    protected void deleteTree(PathFragment path) throws IOException {
        this.deleteTreesBelow(path);
        this.delete(path);
    }

    protected void deleteTreesBelow(PathFragment dir) throws IOException {
        if (this.isDirectory(dir, false)) {
            Collection<String> entries;
            try {
                entries = this.getDirectoryEntries(dir);
            }
            catch (IOException e) {
                this.setReadable(dir, true);
                this.setExecutable(dir, true);
                entries = this.getDirectoryEntries(dir);
            }
            Iterator<String> iterator = entries.iterator();
            if (iterator.hasNext()) {
                PathFragment first = dir.getChild(iterator.next());
                this.deleteTreesBelow(first);
                try {
                    if (!this.delete(first)) {
                        throw new IOException("Unable to delete \"" + first + "\": directory entry does not exist");
                    }
                }
                catch (IOException e) {
                    this.setWritable(dir, true);
                    this.setExecutable(dir, true);
                    this.deleteTreesBelow(first);
                    this.delete(first);
                }
            }
            while (iterator.hasNext()) {
                PathFragment path = dir.getChild(iterator.next());
                this.deleteTreesBelow(path);
                this.delete(path);
            }
        }
    }

    protected abstract long getLastModifiedTime(PathFragment var1, boolean var2) throws IOException;

    public abstract void setLastModifiedTime(PathFragment var1, long var2) throws IOException;

    public byte[] getxattr(PathFragment path, String name, boolean followSymlinks) throws IOException {
        return null;
    }

    protected byte[] getFastDigest(PathFragment path) throws IOException {
        return null;
    }

    protected byte[] getDigest(final PathFragment path) throws IOException {
        return new ByteSource(){

            @Override
            public InputStream openStream() throws IOException {
                return FileSystem.this.getInputStream(path);
            }
        }.hash(this.digestFunction.getHashFunction()).asBytes();
    }

    protected abstract boolean isSymbolicLink(PathFragment var1);

    protected final PathFragment appendSegment(PathFragment dir, String child, int maxLinks) throws IOException {
        PathFragment naive = dir.getChild(child);
        PathFragment linkTarget = this.resolveOneLink(naive);
        if (linkTarget == null) {
            return naive;
        }
        if (maxLinks-- == 0) {
            throw new FileSymlinkLoopException(naive);
        }
        if (linkTarget.isAbsolute()) {
            dir = PathFragment.createAlreadyNormalized(linkTarget.getDriveStr());
        }
        for (String name : linkTarget.segments()) {
            if (name.equals(".") || name.isEmpty()) continue;
            if (name.equals("..")) {
                PathFragment parent = dir.getParentDirectory();
                if (parent == null) continue;
                dir = parent;
                continue;
            }
            dir = this.appendSegment(dir, name, maxLinks);
        }
        return dir;
    }

    @Nullable
    protected PathFragment resolveOneLink(PathFragment path) throws IOException {
        try {
            return this.readSymbolicLink(path);
        }
        catch (NotASymlinkException e) {
            if (!this.exists(path, false)) {
                throw new FileNotFoundException(path + " (No such file or directory)");
            }
            return null;
        }
    }

    protected Path resolveSymbolicLinks(PathFragment path) throws IOException {
        PathFragment parentNode = path.getParentDirectory();
        return parentNode == null ? this.getPath(path) : this.getPath(this.appendSegment(this.resolveSymbolicLinks(parentNode).asFragment(), path.getBaseName(), 32));
    }

    protected FileStatus stat(final PathFragment path, final boolean followSymlinks) throws IOException {
        FileStatus status = new FileStatus(){
            volatile Boolean isFile;
            volatile Boolean isDirectory;
            volatile Boolean isSymbolicLink;
            volatile Boolean isSpecial;
            volatile long size = -1L;
            volatile long mtime = -1L;

            @Override
            public boolean isFile() {
                if (this.isFile == null) {
                    this.isFile = FileSystem.this.isFile(path, followSymlinks);
                }
                return this.isFile;
            }

            @Override
            public boolean isDirectory() {
                if (this.isDirectory == null) {
                    this.isDirectory = FileSystem.this.isDirectory(path, followSymlinks);
                }
                return this.isDirectory;
            }

            @Override
            public boolean isSymbolicLink() {
                if (this.isSymbolicLink == null) {
                    this.isSymbolicLink = FileSystem.this.isSymbolicLink(path);
                }
                return this.isSymbolicLink;
            }

            @Override
            public boolean isSpecialFile() {
                if (this.isSpecial == null) {
                    this.isSpecial = FileSystem.this.isSpecialFile(path, followSymlinks);
                }
                return this.isSpecial;
            }

            @Override
            public long getSize() throws IOException {
                if (this.size == -1L) {
                    this.size = FileSystem.this.getFileSize(path, followSymlinks);
                }
                return this.size;
            }

            @Override
            public long getLastModifiedTime() throws IOException {
                if (this.mtime == -1L) {
                    this.mtime = FileSystem.this.getLastModifiedTime(path, followSymlinks);
                }
                return this.mtime;
            }

            @Override
            public long getLastChangeTime() {
                throw new UnsupportedOperationException();
            }

            @Override
            public long getNodeId() {
                throw new UnsupportedOperationException();
            }
        };
        status.getLastModifiedTime();
        return status;
    }

    @Nullable
    protected FileStatus statNullable(PathFragment path, boolean followSymlinks) {
        try {
            return this.stat(path, followSymlinks);
        }
        catch (IOException e) {
            return null;
        }
    }

    @Nullable
    protected FileStatus statIfFound(PathFragment path, boolean followSymlinks) throws IOException {
        try {
            return this.stat(path, followSymlinks);
        }
        catch (FileNotFoundException e) {
            return null;
        }
    }

    protected abstract boolean isDirectory(PathFragment var1, boolean var2);

    protected abstract boolean isFile(PathFragment var1, boolean var2);

    protected abstract boolean isSpecialFile(PathFragment var1, boolean var2);

    protected abstract void createSymbolicLink(PathFragment var1, PathFragment var2) throws IOException;

    protected abstract PathFragment readSymbolicLink(PathFragment var1) throws IOException;

    protected PathFragment readSymbolicLinkUnchecked(PathFragment path) throws IOException {
        return this.readSymbolicLink(path);
    }

    public boolean exists(PathFragment path) {
        return this.exists(path, true);
    }

    protected abstract boolean exists(PathFragment var1, boolean var2);

    protected abstract Collection<String> getDirectoryEntries(PathFragment var1) throws IOException;

    protected static Dirent.Type direntFromStat(FileStatus stat) {
        if (stat == null) {
            return Dirent.Type.UNKNOWN;
        }
        if (stat.isSpecialFile()) {
            return Dirent.Type.UNKNOWN;
        }
        if (stat.isFile()) {
            return Dirent.Type.FILE;
        }
        if (stat.isDirectory()) {
            return Dirent.Type.DIRECTORY;
        }
        if (stat.isSymbolicLink()) {
            return Dirent.Type.SYMLINK;
        }
        return Dirent.Type.UNKNOWN;
    }

    protected Collection<Dirent> readdir(PathFragment path, boolean followSymlinks) throws IOException {
        Collection<String> children = this.getDirectoryEntries(path);
        ArrayList<Dirent> dirents = Lists.newArrayListWithCapacity(children.size());
        for (String child : children) {
            PathFragment childPath = path.getChild(child);
            Dirent.Type type = FileSystem.direntFromStat(this.statNullable(childPath, followSymlinks));
            dirents.add(new Dirent(child, type));
        }
        return dirents;
    }

    protected abstract boolean isReadable(PathFragment var1) throws IOException;

    protected abstract void setReadable(PathFragment var1, boolean var2) throws IOException;

    protected abstract boolean isWritable(PathFragment var1) throws IOException;

    public abstract void setWritable(PathFragment var1, boolean var2) throws IOException;

    protected abstract boolean isExecutable(PathFragment var1) throws IOException;

    protected abstract void setExecutable(PathFragment var1, boolean var2) throws IOException;

    protected void chmod(PathFragment path, int mode) throws IOException {
        this.setReadable(path, (mode & 0x100) != 0);
        this.setWritable(path, (mode & 0x80) != 0);
        this.setExecutable(path, (mode & 0x40) != 0);
    }

    protected abstract InputStream getInputStream(PathFragment var1) throws IOException;

    protected ReadableByteChannel createReadableByteChannel(PathFragment path) throws IOException {
        throw new UnsupportedOperationException();
    }

    protected abstract SeekableByteChannel createReadWriteByteChannel(PathFragment var1) throws IOException;

    protected final OutputStream getOutputStream(PathFragment path) throws IOException {
        return this.getOutputStream(path, false);
    }

    protected abstract OutputStream getOutputStream(PathFragment var1, boolean var2) throws IOException;

    protected abstract OutputStream getOutputStream(PathFragment var1, boolean var2, boolean var3) throws IOException;

    public abstract void renameTo(PathFragment var1, PathFragment var2) throws IOException;

    protected void createHardLink(PathFragment linkPath, PathFragment originalPath) throws IOException {
        if (!this.exists(originalPath)) {
            throw new FileNotFoundException("File \"" + originalPath.getBaseName() + "\" linked from \"" + linkPath.getBaseName() + "\" does not exist");
        }
        if (this.exists(linkPath)) {
            throw new FileAlreadyExistsException("New link file \"" + linkPath.getBaseName() + "\" already exists");
        }
        this.createFSDependentHardLink(linkPath, originalPath);
    }

    protected abstract void createFSDependentHardLink(PathFragment var1, PathFragment var2) throws IOException;

    protected void prefetchPackageAsync(PathFragment path, int maxDirs) {
    }

    public static final class NotASymlinkException
    extends IOException {
        public NotASymlinkException(PathFragment path) {
            super(path.getPathString() + " is not a symlink");
        }

        public NotASymlinkException(PathFragment path, Throwable cause) {
            super(path.getPathString() + " is not a symlink", cause);
        }
    }
}

