/*
 * Decompiled with CFR 0.152.
 */
package com.google.devtools.build.lib.profiler;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Predicate;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Iterables;
import com.google.devtools.build.lib.actions.ResourceEstimator;
import com.google.devtools.build.lib.bugreport.BugReporter;
import com.google.devtools.build.lib.clock.Clock;
import com.google.devtools.build.lib.collect.Extrema;
import com.google.devtools.build.lib.profiler.CollectLocalResourceUsage;
import com.google.devtools.build.lib.profiler.CounterSeriesTraceData;
import com.google.devtools.build.lib.profiler.JsonTraceFileWriter;
import com.google.devtools.build.lib.profiler.MemoryProfiler;
import com.google.devtools.build.lib.profiler.PredicateBasedStatRecorder;
import com.google.devtools.build.lib.profiler.ProfilePhase;
import com.google.devtools.build.lib.profiler.ProfilerTask;
import com.google.devtools.build.lib.profiler.SilentCloseable;
import com.google.devtools.build.lib.profiler.SingleStatRecorder;
import com.google.devtools.build.lib.profiler.StatRecorder;
import com.google.devtools.build.lib.profiler.TimeSeries;
import com.google.devtools.build.lib.profiler.TraceData;
import com.google.devtools.build.lib.worker.WorkerMetricsCollector;
import com.google.gson.stream.JsonWriter;
import com.sun.management.OperatingSystemMXBean;
import java.io.IOException;
import java.io.OutputStream;
import java.lang.management.ManagementFactory;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.EnumSet;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Supplier;
import java.util.zip.GZIPOutputStream;
import javax.annotation.Nullable;

public final class Profiler {
    private static final Profiler instance = new Profiler();
    private static final int HISTOGRAM_BUCKETS = 20;
    private static final Duration ACTION_COUNT_BUCKET_DURATION = Duration.ofMillis(200L);
    private Clock clock;
    private Set<ProfilerTask> profiledTasks;
    private volatile long profileStartTime;
    private volatile boolean recordAllDurations = false;
    private Duration profileCpuStartTime;
    private final AtomicReference<JsonTraceFileWriter> writerRef = new AtomicReference();
    private final SlowestTaskAggregator[] slowestTasks = new SlowestTaskAggregator[ProfilerTask.values().length];
    @VisibleForTesting
    final StatRecorder[] tasksHistograms = new StatRecorder[ProfilerTask.values().length];
    private CollectLocalResourceUsage resourceUsageThread;
    private TimeSeries actionCountTimeSeries;
    private TimeSeries actionCacheCountTimeSeries;
    private Duration actionCountStartTime;
    private boolean collectTaskHistograms;
    private boolean includePrimaryOutput;
    private boolean includeTargetLabel;
    private static final SilentCloseable NOP = () -> {};

    private Profiler() {
        this.initHistograms();
        for (ProfilerTask task : ProfilerTask.values()) {
            if (!task.collectsSlowestInstances) continue;
            this.slowestTasks[task.ordinal()] = new SlowestTaskAggregator();
        }
    }

    private void initHistograms() {
        for (ProfilerTask task : ProfilerTask.values()) {
            if (task.isVfs()) {
                Map<String, ? extends Predicate<? super String>> vfsHeuristics = StatRecorder.VfsHeuristics.vfsTypeHeuristics;
                ArrayList<PredicateBasedStatRecorder.RecorderAndPredicate> recorders = new ArrayList<PredicateBasedStatRecorder.RecorderAndPredicate>(vfsHeuristics.size());
                for (Map.Entry<String, ? extends Predicate<? super String>> e : vfsHeuristics.entrySet()) {
                    recorders.add(new PredicateBasedStatRecorder.RecorderAndPredicate(new SingleStatRecorder(task + " " + e.getKey(), 20), e.getValue()));
                }
                this.tasksHistograms[task.ordinal()] = new PredicateBasedStatRecorder(recorders);
                continue;
            }
            this.tasksHistograms[task.ordinal()] = new SingleStatRecorder((Object)task, 20);
        }
    }

    public synchronized ImmutableList<StatRecorder> getTasksHistograms() {
        return this.isActive() ? ImmutableList.copyOf(this.tasksHistograms) : ImmutableList.of();
    }

    public static Profiler instance() {
        return instance;
    }

    public static long nanoTimeMaybe() {
        if (instance.isActive()) {
            return Profiler.instance.clock.nanoTime();
        }
        return -1L;
    }

    @Nullable
    public static Duration elapsedTimeMaybe() {
        if (instance.isActive()) {
            return Duration.ofNanos(Profiler.instance.clock.nanoTime()).minus(Duration.ofNanos(Profiler.instance.profileStartTime));
        }
        return null;
    }

    private static Duration getProcessCpuTime() {
        OperatingSystemMXBean bean = (OperatingSystemMXBean)ManagementFactory.getOperatingSystemMXBean();
        return Duration.ofNanos(bean.getProcessCpuTime());
    }

    @Nullable
    public static Duration getProcessCpuTimeMaybe() {
        if (Profiler.instance().isActive()) {
            return Profiler.getProcessCpuTime().minus(Profiler.instance().profileCpuStartTime);
        }
        return null;
    }

    public synchronized void start(ImmutableSet<ProfilerTask> profiledTasks, OutputStream stream, Format format, String outputBase, UUID buildID, boolean recordAllDurations, Clock clock, long execStartTimeNanos, boolean slimProfile, boolean includePrimaryOutput, boolean includeTargetLabel, boolean collectTaskHistograms, boolean collectWorkerDataInProfiler, boolean collectLoadAverage, boolean collectSystemNetworkUsage, boolean collectPressureStallIndicators, boolean collectResourceEstimation, ResourceEstimator resourceEstimator, WorkerMetricsCollector workerMetricsCollector, BugReporter bugReporter) throws IOException {
        Preconditions.checkState(!this.isActive(), "Profiler already active");
        this.initHistograms();
        this.profiledTasks = profiledTasks.isEmpty() ? profiledTasks : EnumSet.copyOf(profiledTasks);
        this.clock = clock;
        this.actionCountStartTime = Duration.ofNanos(clock.nanoTime());
        this.actionCountTimeSeries = new TimeSeries(this.actionCountStartTime, ACTION_COUNT_BUCKET_DURATION);
        this.actionCacheCountTimeSeries = new TimeSeries(this.actionCountStartTime, ACTION_COUNT_BUCKET_DURATION);
        this.collectTaskHistograms = collectTaskHistograms;
        this.includePrimaryOutput = includePrimaryOutput;
        this.includeTargetLabel = includeTargetLabel;
        this.recordAllDurations = recordAllDurations;
        JsonTraceFileWriter writer = null;
        if (stream != null && format != null) {
            switch (format) {
                case JSON_TRACE_FILE_FORMAT: {
                    writer = new JsonTraceFileWriter(stream, execStartTimeNanos, slimProfile, outputBase, buildID);
                    break;
                }
                case JSON_TRACE_FILE_COMPRESSED_FORMAT: {
                    writer = new JsonTraceFileWriter(new GZIPOutputStream(stream), execStartTimeNanos, slimProfile, outputBase, buildID);
                }
            }
            writer.start();
        }
        this.writerRef.set(writer);
        this.profileStartTime = execStartTimeNanos;
        this.profileCpuStartTime = Profiler.getProcessCpuTime();
        this.resourceUsageThread = new CollectLocalResourceUsage(bugReporter, workerMetricsCollector, resourceEstimator, collectWorkerDataInProfiler, collectLoadAverage, collectSystemNetworkUsage, collectResourceEstimation, collectPressureStallIndicators);
        this.resourceUsageThread.setDaemon(true);
        this.resourceUsageThread.start();
    }

    public synchronized Iterable<SlowTask> getSlowestTasks() {
        ArrayList<ImmutableList<SlowTask>> slowestTasksByType = new ArrayList<ImmutableList<SlowTask>>();
        for (SlowestTaskAggregator aggregator : this.slowestTasks) {
            if (aggregator == null) continue;
            slowestTasksByType.add(aggregator.getSlowestTasks());
        }
        return Iterables.concat(slowestTasksByType);
    }

    private void collectActionCounts() {
        Duration endTime = Duration.ofNanos(this.clock.nanoTime());
        int len = (int)endTime.minus(this.actionCountStartTime).dividedBy(ACTION_COUNT_BUCKET_DURATION) + 1;
        LinkedHashMap<ProfilerTask, double[]> counterSeriesMap = new LinkedHashMap<ProfilerTask, double[]>();
        if (this.actionCountTimeSeries != null) {
            double[] actionCountValues = this.actionCountTimeSeries.toDoubleArray(len);
            this.actionCountTimeSeries = null;
            counterSeriesMap.put(ProfilerTask.ACTION_COUNTS, actionCountValues);
        }
        if (this.actionCacheCountTimeSeries != null) {
            double[] actionCacheCountValues = this.actionCacheCountTimeSeries.toDoubleArray(len);
            this.actionCacheCountTimeSeries = null;
            counterSeriesMap.put(ProfilerTask.ACTION_CACHE_COUNTS, actionCacheCountValues);
        }
        if (!counterSeriesMap.isEmpty()) {
            instance.logCounters(counterSeriesMap, this.actionCountStartTime, ACTION_COUNT_BUCKET_DURATION);
        }
    }

    public synchronized void stop() throws IOException {
        if (!this.isActive()) {
            return;
        }
        this.collectActionCounts();
        if (this.resourceUsageThread != null) {
            this.resourceUsageThread.stopCollecting();
            try {
                this.resourceUsageThread.join();
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
            this.resourceUsageThread.logCollectedData();
            this.resourceUsageThread = null;
        }
        this.logEvent(ProfilerTask.INFO, "Finishing");
        JsonTraceFileWriter writer = this.writerRef.getAndSet(null);
        if (writer != null) {
            writer.shutdown();
            writer = null;
        }
        Arrays.fill(this.tasksHistograms, null);
        this.profileStartTime = 0L;
        this.profileCpuStartTime = null;
        for (SlowestTaskAggregator aggregator : this.slowestTasks) {
            if (aggregator == null) continue;
            aggregator.clear();
        }
    }

    public boolean isActive() {
        return this.profileStartTime != 0L;
    }

    public boolean isProfiling(ProfilerTask type) {
        return this.profiledTasks.contains((Object)type);
    }

    private boolean wasTaskSlowEnoughToRecord(ProfilerTask type, long duration) {
        return this.recordAllDurations || duration >= type.minDuration;
    }

    public void logCounters(Map<ProfilerTask, double[]> counterSeriesMap, Duration profileStart, Duration bucketDuration) {
        JsonTraceFileWriter currentWriter = this.writerRef.get();
        if (this.isActive() && currentWriter != null) {
            CounterSeriesTraceData counterSeriesTraceData = new CounterSeriesTraceData(counterSeriesMap, profileStart, bucketDuration);
            currentWriter.enqueue(counterSeriesTraceData);
        }
    }

    private void logTask(long startTimeNanos, long duration, ProfilerTask type, String description) {
        Preconditions.checkNotNull(description);
        Preconditions.checkState(!"".equals(description), "No description -> not helpful");
        if (duration < 0L) {
            duration = 0L;
        }
        StatRecorder statRecorder = this.tasksHistograms[type.ordinal()];
        if (this.collectTaskHistograms && statRecorder != null) {
            statRecorder.addStat((int)Duration.ofNanos(duration).toMillis(), description);
        }
        if (this.isActive() && startTimeNanos >= 0L && this.isProfiling(type)) {
            JsonTraceFileWriter currentWriter = this.writerRef.get();
            if (this.wasTaskSlowEnoughToRecord(type, duration)) {
                SlowestTaskAggregator aggregator;
                TaskData data = new TaskData(startTimeNanos, type, description);
                data.durationNanos = duration;
                if (currentWriter != null) {
                    currentWriter.enqueue(data);
                }
                if ((aggregator = this.slowestTasks[type.ordinal()]) != null) {
                    aggregator.add(data);
                }
            }
        }
    }

    public void logSimpleTask(long startTimeNanos, ProfilerTask type, String description) {
        if (this.clock != null) {
            this.logTask(startTimeNanos, this.clock.nanoTime() - startTimeNanos, type, description);
        }
    }

    public void logSimpleTask(long startTimeNanos, long stopTimeNanos, ProfilerTask type, String description) {
        this.logTask(startTimeNanos, stopTimeNanos - startTimeNanos, type, description);
    }

    public void logSimpleTaskDuration(long startTimeNanos, Duration duration, ProfilerTask type, String description) {
        this.logTask(startTimeNanos, duration.toNanos(), type, description);
    }

    public void logEventAtTime(long atTimeNanos, ProfilerTask type, String description) {
        this.logTask(atTimeNanos, 0L, type, description);
    }

    @VisibleForTesting
    void logEvent(ProfilerTask type, String description) {
        this.logEventAtTime(this.clock.nanoTime(), type, description);
    }

    private SilentCloseable reallyProfile(ProfilerTask type, String description) {
        long startTimeNanos = this.clock.nanoTime();
        return () -> this.completeTask(startTimeNanos, type, description);
    }

    public SilentCloseable profile(ProfilerTask type, String description) {
        return this.isActive() && this.isProfiling(type) ? this.reallyProfile(type, description) : NOP;
    }

    public SilentCloseable profile(ProfilerTask type, Supplier<String> description) {
        return this.isActive() && this.isProfiling(type) ? this.reallyProfile(type, description.get()) : NOP;
    }

    public SilentCloseable profile(String description) {
        return this.profile(ProfilerTask.INFO, description);
    }

    public SilentCloseable profileAction(ProfilerTask type, String mnemonic, String description, String primaryOutput, String targetLabel) {
        Preconditions.checkNotNull(description);
        if (this.isActive() && this.isProfiling(type)) {
            long startTimeNanos = this.clock.nanoTime();
            return () -> this.completeAction(startTimeNanos, type, description, mnemonic, this.includePrimaryOutput ? primaryOutput : null, this.includeTargetLabel ? targetLabel : null);
        }
        return NOP;
    }

    public SilentCloseable profileAction(ProfilerTask type, String description, String primaryOutput, String targetLabel) {
        return this.profileAction(type, null, description, primaryOutput, targetLabel);
    }

    private boolean countAction(ProfilerTask type, TaskData taskData) {
        return type == ProfilerTask.ACTION || type == ProfilerTask.INFO && "discoverInputs".equals(taskData.description);
    }

    private void completeTask(long startTimeNanos, ProfilerTask type, String description) {
        long endTimeNanos;
        long duration;
        boolean shouldRecordTask;
        if (this.isActive() && (shouldRecordTask = this.wasTaskSlowEnoughToRecord(type, duration = (endTimeNanos = this.clock.nanoTime()) - startTimeNanos))) {
            this.recordTask(new TaskData(startTimeNanos, duration, type, description));
        }
    }

    private void completeAction(long startTimeNanos, ProfilerTask type, String description, String mnemonic, @Nullable String primaryOutput, @Nullable String targetLabel) {
        long endTimeNanos;
        long duration;
        boolean shouldRecordTask;
        if (this.isActive() && (shouldRecordTask = this.wasTaskSlowEnoughToRecord(type, duration = (endTimeNanos = this.clock.nanoTime()) - startTimeNanos))) {
            this.recordTask(new ActionTaskData(startTimeNanos, duration, type, mnemonic, description, primaryOutput, targetLabel));
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void recordTask(TaskData data) {
        SlowestTaskAggregator aggregator;
        Profiler profiler;
        JsonTraceFileWriter writer = this.writerRef.get();
        if (writer != null) {
            writer.enqueue(data);
        }
        long endTimeNanos = data.startTimeNanos + data.durationNanos;
        if (this.actionCountTimeSeries != null && this.countAction(data.type, data)) {
            profiler = this;
            synchronized (profiler) {
                this.actionCountTimeSeries.addRange(Duration.ofNanos(data.startTimeNanos), Duration.ofNanos(endTimeNanos));
            }
        }
        if (this.actionCacheCountTimeSeries != null && data.type == ProfilerTask.ACTION_CHECK) {
            profiler = this;
            synchronized (profiler) {
                this.actionCacheCountTimeSeries.addRange(Duration.ofNanos(data.startTimeNanos), Duration.ofNanos(endTimeNanos));
            }
        }
        if ((aggregator = this.slowestTasks[data.type.ordinal()]) != null) {
            aggregator.add(data);
        }
    }

    public void markPhase(ProfilePhase phase) throws InterruptedException {
        MemoryProfiler.instance().markPhase(phase);
        if (this.isActive() && this.isProfiling(ProfilerTask.PHASE)) {
            this.logEvent(ProfilerTask.PHASE, phase.description);
        }
    }

    private static final class SlowestTaskAggregator {
        private static final int SHARDS = 16;
        private static final int SIZE = 30;
        private final Extrema<SlowTask>[] extremaAggregators = new Extrema[16];

        SlowestTaskAggregator() {
            for (int i = 0; i < 16; ++i) {
                this.extremaAggregators[i] = Extrema.max(30);
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        void add(TaskData taskData) {
            Extrema<SlowTask> extrema;
            Extrema<SlowTask> extrema2 = extrema = this.extremaAggregators[(int)(Thread.currentThread().getId() % 16L)];
            synchronized (extrema2) {
                extrema.aggregate(new SlowTask(taskData));
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        void clear() {
            for (int i = 0; i < 16; ++i) {
                Extrema<SlowTask> extrema;
                Extrema<SlowTask> extrema2 = extrema = this.extremaAggregators[i];
                synchronized (extrema2) {
                    extrema.clear();
                    continue;
                }
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        ImmutableList<SlowTask> getSlowestTasks() {
            Extrema<SlowTask> mergedExtrema = Extrema.max(30);
            for (int i = 0; i < 16; ++i) {
                Extrema<SlowTask> extrema;
                Extrema<SlowTask> extrema2 = extrema = this.extremaAggregators[i];
                synchronized (extrema2) {
                    for (SlowTask task : extrema.getExtremeElements()) {
                        mergedExtrema.aggregate(task);
                    }
                    continue;
                }
            }
            return mergedExtrema.getExtremeElements();
        }
    }

    static final class ActionTaskData
    extends TaskData {
        @Nullable
        final String primaryOutputPath;
        @Nullable
        final String targetLabel;
        @Nullable
        final String mnemonic;

        ActionTaskData(long startTimeNanos, long durationNanos, ProfilerTask eventType, @Nullable String mnemonic, String description, @Nullable String primaryOutputPath, @Nullable String targetLabel) {
            super(startTimeNanos, durationNanos, eventType, description);
            this.primaryOutputPath = primaryOutputPath;
            this.targetLabel = targetLabel;
            this.mnemonic = mnemonic;
        }
    }

    static class TaskData
    implements TraceData {
        final long threadId;
        final long startTimeNanos;
        final ProfilerTask type;
        final String description;
        long durationNanos;

        TaskData(long startTimeNanos, long durationNanos, ProfilerTask eventType, String description) {
            this.threadId = Thread.currentThread().getId();
            this.startTimeNanos = startTimeNanos;
            this.durationNanos = durationNanos;
            this.type = eventType;
            this.description = Preconditions.checkNotNull(description);
        }

        TaskData(long startTimeNanos, ProfilerTask eventType, String description) {
            this(startTimeNanos, -1L, eventType, description);
        }

        TaskData(long threadId, long startTimeNanos, long durationNanos, String description) {
            this.type = ProfilerTask.UNKNOWN;
            this.threadId = threadId;
            this.startTimeNanos = startTimeNanos;
            this.durationNanos = durationNanos;
            this.description = description;
        }

        public String toString() {
            return "Thread " + this.threadId + ", type " + this.type + ", " + this.description;
        }

        @Override
        public void writeTraceData(JsonWriter jsonWriter, long profileStartTimeNanos) throws IOException {
            String eventType = this.durationNanos == 0L ? "i" : "X";
            jsonWriter.setIndent("  ");
            jsonWriter.beginObject();
            jsonWriter.setIndent("");
            if (this.type == null) {
                jsonWriter.setIndent("    ");
            } else {
                jsonWriter.name("cat").value(this.type.description);
            }
            jsonWriter.name("name").value(this.description);
            jsonWriter.name("ph").value(eventType);
            jsonWriter.name("ts").value(TimeUnit.NANOSECONDS.toMicros(this.startTimeNanos - profileStartTimeNanos));
            if (this.durationNanos != 0L) {
                jsonWriter.name("dur").value(TimeUnit.NANOSECONDS.toMicros(this.durationNanos));
            }
            jsonWriter.name("pid").value(1L);
            if (this instanceof ActionTaskData) {
                ActionTaskData actionTaskData = (ActionTaskData)this;
                if (actionTaskData.primaryOutputPath != null) {
                    jsonWriter.name("out").value(actionTaskData.primaryOutputPath);
                }
                if (actionTaskData.targetLabel != null || actionTaskData.mnemonic != null) {
                    jsonWriter.name("args");
                    jsonWriter.beginObject();
                    if (actionTaskData.targetLabel != null) {
                        jsonWriter.name("target").value(actionTaskData.targetLabel);
                    }
                    if (actionTaskData.mnemonic != null) {
                        jsonWriter.name("mnemonic").value(actionTaskData.mnemonic);
                    }
                    jsonWriter.endObject();
                }
            }
            if (this.type == ProfilerTask.CRITICAL_PATH_COMPONENT) {
                jsonWriter.name("args");
                jsonWriter.beginObject();
                jsonWriter.name("tid").value(this.threadId);
                jsonWriter.endObject();
            }
            jsonWriter.name("tid").value(this.type == ProfilerTask.CRITICAL_PATH_COMPONENT ? 0L : this.threadId);
            jsonWriter.endObject();
        }
    }

    public static final class SlowTask
    implements Comparable<SlowTask> {
        final long durationNanos;
        final String description;
        final ProfilerTask type;

        private SlowTask(TaskData taskData) {
            this.durationNanos = taskData.durationNanos;
            this.description = taskData.description;
            this.type = taskData.type;
        }

        @Override
        public int compareTo(SlowTask other) {
            long delta = this.durationNanos - other.durationNanos;
            if (delta < 0L) {
                return -1;
            }
            if (delta > 0L) {
                return 1;
            }
            return 0;
        }

        public long getDurationNanos() {
            return this.durationNanos;
        }

        public String getDescription() {
            return this.description;
        }

        public ProfilerTask getType() {
            return this.type;
        }
    }

    public static enum Format {
        JSON_TRACE_FILE_FORMAT,
        JSON_TRACE_FILE_COMPRESSED_FORMAT;

    }
}

