/*
 * Decompiled with CFR 0.152.
 */
package com.google.devtools.build.lib.profiler;

import com.google.common.base.Preconditions;
import com.google.common.base.Stopwatch;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.devtools.build.lib.actions.ResourceEstimator;
import com.google.devtools.build.lib.bugreport.BugReporter;
import com.google.devtools.build.lib.profiler.NetworkMetricsCollector;
import com.google.devtools.build.lib.profiler.Profiler;
import com.google.devtools.build.lib.profiler.ProfilerTask;
import com.google.devtools.build.lib.profiler.SilentCloseable;
import com.google.devtools.build.lib.profiler.TimeSeries;
import com.google.devtools.build.lib.unix.ProcMeminfoParser;
import com.google.devtools.build.lib.util.OS;
import com.google.devtools.build.lib.util.ResourceUsage;
import com.google.devtools.build.lib.worker.WorkerMetric;
import com.google.devtools.build.lib.worker.WorkerMetricsCollector;
import com.google.errorprone.annotations.concurrent.GuardedBy;
import com.sun.management.OperatingSystemMXBean;
import java.io.IOException;
import java.lang.management.ManagementFactory;
import java.lang.management.MemoryMXBean;
import java.time.Duration;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import javax.annotation.Nullable;

public class CollectLocalResourceUsage
extends Thread {
    private static final Duration BUCKET_DURATION = Duration.ofSeconds(1L);
    private static final Duration LOCAL_RESOURCES_COLLECT_SLEEP_INTERVAL = Duration.ofMillis(200L);
    private final BugReporter bugReporter;
    private final boolean collectWorkerDataInProfiler;
    private final boolean collectLoadAverage;
    private final boolean collectSystemNetworkUsage;
    private final boolean collectResourceManagerEstimation;
    private volatile boolean stopLocalUsageCollection;
    private volatile boolean profilingStarted;
    @Nullable
    @GuardedBy(value="this")
    private Map<ProfilerTask, TimeSeries> timeSeries;
    private Stopwatch stopwatch;
    private final WorkerMetricsCollector workerMetricsCollector;
    private final ResourceEstimator resourceEstimator;
    private final boolean collectPressureStallIndicators;

    CollectLocalResourceUsage(BugReporter bugReporter, WorkerMetricsCollector workerMetricsCollector, ResourceEstimator resourceEstimator, boolean collectWorkerDataInProfiler, boolean collectLoadAverage, boolean collectSystemNetworkUsage, boolean collectResourceManagerEstimation, boolean collectPressureStallIndicators) {
        super("collect-local-resources");
        this.bugReporter = Preconditions.checkNotNull(bugReporter);
        this.collectWorkerDataInProfiler = collectWorkerDataInProfiler;
        this.workerMetricsCollector = workerMetricsCollector;
        this.collectLoadAverage = collectLoadAverage;
        this.collectSystemNetworkUsage = collectSystemNetworkUsage;
        this.collectResourceManagerEstimation = collectResourceManagerEstimation;
        this.resourceEstimator = resourceEstimator;
        this.collectPressureStallIndicators = collectPressureStallIndicators;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void run() {
        int numProcessors = Runtime.getRuntime().availableProcessors();
        this.stopwatch = Stopwatch.createStarted();
        CollectLocalResourceUsage collectLocalResourceUsage = this;
        synchronized (collectLocalResourceUsage) {
            this.timeSeries = new HashMap<ProfilerTask, TimeSeries>();
            Duration startTime = this.stopwatch.elapsed();
            ArrayList<ProfilerTask> enabledCounters = new ArrayList<ProfilerTask>();
            enabledCounters.addAll(ImmutableList.of(ProfilerTask.LOCAL_CPU_USAGE, ProfilerTask.LOCAL_MEMORY_USAGE, ProfilerTask.SYSTEM_CPU_USAGE, ProfilerTask.SYSTEM_MEMORY_USAGE));
            if (this.collectWorkerDataInProfiler) {
                enabledCounters.add(ProfilerTask.WORKERS_MEMORY_USAGE);
            }
            if (this.collectLoadAverage) {
                enabledCounters.add(ProfilerTask.SYSTEM_LOAD_AVERAGE);
            }
            if (this.collectSystemNetworkUsage) {
                enabledCounters.add(ProfilerTask.SYSTEM_NETWORK_UP_USAGE);
                enabledCounters.add(ProfilerTask.SYSTEM_NETWORK_DOWN_USAGE);
            }
            if (this.collectResourceManagerEstimation) {
                enabledCounters.add(ProfilerTask.MEMORY_USAGE_ESTIMATION);
                enabledCounters.add(ProfilerTask.CPU_USAGE_ESTIMATION);
            }
            if (this.collectPressureStallIndicators) {
                enabledCounters.add(ProfilerTask.PRESSURE_STALL_IO);
                enabledCounters.add(ProfilerTask.PRESSURE_STALL_MEMORY);
            }
            for (ProfilerTask counter : enabledCounters) {
                this.timeSeries.put(counter, new TimeSeries(startTime, BUCKET_DURATION));
            }
        }
        OperatingSystemMXBean osBean = (OperatingSystemMXBean)ManagementFactory.getOperatingSystemMXBean();
        MemoryMXBean memoryBean = ManagementFactory.getMemoryMXBean();
        Duration previousElapsed = this.stopwatch.elapsed();
        long previousCpuTimeNanos = osBean.getProcessCpuTime();
        this.profilingStarted = true;
        while (!this.stopLocalUsageCollection) {
            long memoryUsage;
            try {
                Thread.sleep(LOCAL_RESOURCES_COLLECT_SLEEP_INTERVAL.toMillis());
            }
            catch (InterruptedException e) {
                return;
            }
            Duration nextElapsed = this.stopwatch.elapsed();
            long nextCpuTimeNanos = osBean.getProcessCpuTime();
            double systemCpuLoad = osBean.getSystemCpuLoad();
            if (Double.isNaN(systemCpuLoad)) {
                systemCpuLoad = 0.0;
            }
            double systemUsage = systemCpuLoad * (double)numProcessors;
            long systemMemoryUsageMb = -1L;
            if (OS.getCurrent() == OS.LINUX) {
                try {
                    ProcMeminfoParser procMeminfoParser = new ProcMeminfoParser("/proc/meminfo");
                    systemMemoryUsageMb = (procMeminfoParser.getTotalKb() - procMeminfoParser.getFreeRamKb()) / 1024L;
                }
                catch (IOException procMeminfoParser) {
                    // empty catch block
                }
            }
            if (systemMemoryUsageMb <= 0L) {
                systemMemoryUsageMb = (osBean.getTotalPhysicalMemorySize() - osBean.getFreePhysicalMemorySize()) / 0x100000L;
            }
            try {
                memoryUsage = memoryBean.getHeapMemoryUsage().getUsed() + memoryBean.getNonHeapMemoryUsage().getUsed();
            }
            catch (IllegalArgumentException e) {
                this.bugReporter.sendBugReport(e);
                memoryUsage = -1L;
            }
            int workerMemoryUsageMb = 0;
            if (this.collectWorkerDataInProfiler) {
                try (SilentCloseable c = Profiler.instance().profile("Worker metrics collection");){
                    workerMemoryUsageMb = this.workerMetricsCollector.collectMetrics().stream().map(WorkerMetric::getWorkerStat).filter(Objects::nonNull).mapToInt(WorkerMetric.WorkerStat::getUsedMemoryInKB).sum() / 1024;
                }
            }
            double loadAverage = 0.0;
            if (this.collectLoadAverage) {
                loadAverage = osBean.getSystemLoadAverage();
            }
            double pressureStallIo = 0.0;
            double pressureStallMemory = 0.0;
            if (this.collectPressureStallIndicators) {
                pressureStallIo = ResourceUsage.readPressureStallIndicator("io");
                pressureStallMemory = ResourceUsage.readPressureStallIndicator("memory");
            }
            double deltaNanos = nextElapsed.minus(previousElapsed).toNanos();
            double cpuLevel = (double)(nextCpuTimeNanos - previousCpuTimeNanos) / deltaNanos;
            NetworkMetricsCollector.SystemNetworkUsages systemNetworkUsages = null;
            if (this.collectSystemNetworkUsage) {
                systemNetworkUsages = NetworkMetricsCollector.instance().collectSystemNetworkUsages(deltaNanos);
            }
            double estimatedCpuUsage = 0.0;
            double estimatedMemoryUsageInMb = 0.0;
            if (this.collectResourceManagerEstimation) {
                estimatedCpuUsage = this.resourceEstimator.getUsedCPU();
                estimatedMemoryUsageInMb = this.resourceEstimator.getUsedMemoryInMb();
            }
            CollectLocalResourceUsage collectLocalResourceUsage2 = this;
            synchronized (collectLocalResourceUsage2) {
                this.addRange(ProfilerTask.LOCAL_CPU_USAGE, previousElapsed, nextElapsed, cpuLevel);
                if (memoryUsage != -1L) {
                    double memoryUsageMb = (double)memoryUsage / 1048576.0;
                    this.addRange(ProfilerTask.LOCAL_MEMORY_USAGE, previousElapsed, nextElapsed, memoryUsageMb);
                }
                this.addRange(ProfilerTask.SYSTEM_CPU_USAGE, previousElapsed, nextElapsed, systemUsage);
                this.addRange(ProfilerTask.SYSTEM_MEMORY_USAGE, previousElapsed, nextElapsed, systemMemoryUsageMb);
                this.addRange(ProfilerTask.WORKERS_MEMORY_USAGE, previousElapsed, nextElapsed, workerMemoryUsageMb);
                if (loadAverage > 0.0) {
                    this.addRange(ProfilerTask.SYSTEM_LOAD_AVERAGE, previousElapsed, nextElapsed, loadAverage);
                }
                if (pressureStallIo >= 0.0) {
                    this.addRange(ProfilerTask.PRESSURE_STALL_IO, previousElapsed, nextElapsed, pressureStallIo);
                }
                if (pressureStallMemory >= 0.0) {
                    this.addRange(ProfilerTask.PRESSURE_STALL_MEMORY, previousElapsed, nextElapsed, pressureStallMemory);
                }
                if (systemNetworkUsages != null) {
                    this.addRange(ProfilerTask.SYSTEM_NETWORK_UP_USAGE, previousElapsed, nextElapsed, systemNetworkUsages.megabitsSentPerSec());
                    this.addRange(ProfilerTask.SYSTEM_NETWORK_DOWN_USAGE, previousElapsed, nextElapsed, systemNetworkUsages.megabitsRecvPerSec());
                }
                this.addRange(ProfilerTask.MEMORY_USAGE_ESTIMATION, previousElapsed, nextElapsed, estimatedMemoryUsageInMb);
                this.addRange(ProfilerTask.CPU_USAGE_ESTIMATION, previousElapsed, nextElapsed, estimatedCpuUsage);
            }
            previousElapsed = nextElapsed;
            previousCpuTimeNanos = nextCpuTimeNanos;
        }
    }

    public void stopCollecting() {
        Preconditions.checkArgument(!this.stopLocalUsageCollection);
        this.stopLocalUsageCollection = true;
        this.interrupt();
    }

    synchronized void logCollectedData() {
        if (!this.profilingStarted) {
            return;
        }
        Preconditions.checkArgument(this.stopLocalUsageCollection);
        long endTimeNanos = System.nanoTime();
        long elapsedNanos = this.stopwatch.elapsed(TimeUnit.NANOSECONDS);
        long startTimeNanos = endTimeNanos - elapsedNanos;
        Duration profileStart = Duration.ofNanos(startTimeNanos);
        int len = (int)(elapsedNanos / BUCKET_DURATION.toNanos()) + 1;
        Profiler profiler = Profiler.instance();
        for (ProfilerTask task : this.timeSeries.keySet()) {
            profiler.logCounters(ImmutableMap.ofEntries(Map.entry(task, this.timeSeries.get((Object)task).toDoubleArray(len))), profileStart, BUCKET_DURATION);
        }
        this.timeSeries = null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void addRange(ProfilerTask type, Duration previousElapsed, Duration nextElapsed, double value) {
        CollectLocalResourceUsage collectLocalResourceUsage = this;
        synchronized (collectLocalResourceUsage) {
            if (this.timeSeries == null) {
                return;
            }
            TimeSeries series = this.timeSeries.get((Object)type);
            if (series != null) {
                series.addRange(previousElapsed, nextElapsed, value);
            }
        }
    }
}

