/*
 * Decompiled with CFR 0.152.
 */
package com.google.devtools.build.lib.packages;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Throwables;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Lists;
import com.google.common.util.concurrent.SettableFuture;
import com.google.devtools.build.lib.actions.ThreadStateReceiver;
import com.google.devtools.build.lib.cmdline.PackageIdentifier;
import com.google.devtools.build.lib.packages.CachingPackageLocator;
import com.google.devtools.build.lib.packages.Globber;
import com.google.devtools.build.lib.packages.GlobberUtils;
import com.google.devtools.build.lib.profiler.SilentCloseable;
import com.google.devtools.build.lib.util.Pair;
import com.google.devtools.build.lib.vfs.Path;
import com.google.devtools.build.lib.vfs.PathFragment;
import com.google.devtools.build.lib.vfs.SyscallCache;
import com.google.devtools.build.lib.vfs.UnixGlob;
import com.google.devtools.build.lib.vfs.UnixGlobPathDiscriminator;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CancellationException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executor;
import java.util.concurrent.Future;
import java.util.concurrent.atomic.AtomicBoolean;

public class GlobCache {
    private final Map<Pair<String, Globber.Operation>, Future<List<Path>>> globCache = new HashMap<Pair<String, Globber.Operation>, Future<List<Path>>>();
    private final Path packageDirectory;
    private final PackageIdentifier packageId;
    private final SyscallCache syscallCache;
    private final int maxDirectoriesToEagerlyVisit;
    private final Executor globExecutor;
    private final AtomicBoolean globalStarted = new AtomicBoolean(false);
    private final CachingPackageLocator packageLocator;
    private final ImmutableSet<PathFragment> ignoredGlobPrefixes;

    public GlobCache(Path packageDirectory, PackageIdentifier packageId, ImmutableSet<PathFragment> ignoredGlobPrefixes, CachingPackageLocator locator, SyscallCache syscallCache, Executor globExecutor, int maxDirectoriesToEagerlyVisit, ThreadStateReceiver threadStateReceiverForMetrics) {
        this.packageDirectory = Preconditions.checkNotNull(packageDirectory);
        this.packageId = Preconditions.checkNotNull(packageId);
        Preconditions.checkNotNull(globExecutor);
        this.globExecutor = command -> globExecutor.execute(() -> {
            try (SilentCloseable ignored = threadStateReceiverForMetrics.started();){
                command.run();
            }
        });
        this.syscallCache = syscallCache;
        this.maxDirectoriesToEagerlyVisit = maxDirectoriesToEagerlyVisit;
        Preconditions.checkNotNull(locator);
        this.packageLocator = locator;
        this.ignoredGlobPrefixes = ignoredGlobPrefixes;
    }

    private boolean globCacheShouldTraverseDirectory(Path directory) {
        if (directory.equals(this.packageDirectory)) {
            return true;
        }
        PathFragment subPackagePath = this.packageId.getPackageFragment().getRelative(directory.relativeTo(this.packageDirectory));
        for (PathFragment ignoredPrefix : this.ignoredGlobPrefixes) {
            if (!subPackagePath.startsWith(ignoredPrefix)) continue;
            return false;
        }
        return !this.isSubPackage(PackageIdentifier.create(this.packageId.getRepository(), subPackagePath));
    }

    private boolean isSubPackage(Path directory) {
        return this.isSubPackage(PackageIdentifier.create(this.packageId.getRepository(), this.packageId.getPackageFragment().getRelative(directory.relativeTo(this.packageDirectory))));
    }

    private boolean isSubPackage(PackageIdentifier subPackageId) {
        return this.packageLocator.getBuildFileForPackage(subPackageId) != null;
    }

    Future<List<Path>> getGlobUnsortedAsync(String pattern, Globber.Operation globberOperation) throws Globber.BadGlobException {
        Future<List<Path>> cached = this.globCache.get(Pair.of(pattern, globberOperation));
        if (cached == null) {
            if (this.maxDirectoriesToEagerlyVisit > -1 && !this.globalStarted.getAndSet(true)) {
                this.packageDirectory.prefetchPackageAsync(this.maxDirectoriesToEagerlyVisit);
            }
            cached = this.safeGlobUnsorted(pattern, globberOperation);
            this.setGlobPaths(pattern, globberOperation, cached);
        }
        return cached;
    }

    @VisibleForTesting
    List<String> getGlobUnsorted(String pattern) throws IOException, Globber.BadGlobException, InterruptedException {
        return this.getGlobUnsorted(pattern, Globber.Operation.FILES_AND_DIRS);
    }

    @VisibleForTesting
    protected List<String> getGlobUnsorted(String pattern, Globber.Operation globberOperation) throws IOException, Globber.BadGlobException, InterruptedException {
        Future<List<Path>> futureResult = this.getGlobUnsortedAsync(pattern, globberOperation);
        List<Path> globPaths = GlobCache.fromFuture(futureResult);
        if (!(futureResult instanceof SettableFuture)) {
            SettableFuture<List<Path>> completedFuture = SettableFuture.create();
            completedFuture.set(globPaths);
            this.globCache.put(Pair.of(pattern, globberOperation), completedFuture);
        }
        ArrayList<String> result = Lists.newArrayListWithCapacity(globPaths.size());
        for (Path path : globPaths) {
            String relative = path.relativeTo(this.packageDirectory).getPathString();
            if (relative.isEmpty()) continue;
            result.add(relative);
        }
        return result;
    }

    private void setGlobPaths(String pattern, Globber.Operation globberOperation, Future<List<Path>> result) {
        this.globCache.put(Pair.of(pattern, globberOperation), result);
    }

    @VisibleForTesting
    Future<List<Path>> safeGlobUnsorted(String pattern, Globber.Operation globberOperation) throws Globber.BadGlobException {
        if (pattern.indexOf(63) != -1) {
            throw new Globber.BadGlobException("glob pattern '" + pattern + "' contains forbidden '?' wildcard");
        }
        String error = UnixGlob.checkPatternForError(pattern);
        if (error != null) {
            throw new Globber.BadGlobException(error + " (in glob pattern '" + pattern + "')");
        }
        try {
            return new UnixGlob.Builder(this.packageDirectory, this.syscallCache).addPattern(pattern).setPathDiscriminator(new GlobUnixPathDiscriminator(globberOperation)).setExecutor(this.globExecutor).globAsync();
        }
        catch (UnixGlob.BadPattern ex) {
            throw new Globber.BadGlobException(ex.getMessage());
        }
    }

    private static List<Path> fromFuture(Future<List<Path>> future) throws IOException, InterruptedException {
        try {
            return future.get();
        }
        catch (ExecutionException e) {
            Throwable cause = e.getCause();
            Throwables.propagateIfPossible(cause, IOException.class, InterruptedException.class);
            throw new RuntimeException(e);
        }
    }

    public List<String> globUnsorted(List<String> includes, List<String> excludes, Globber.Operation globberOperation, boolean allowEmpty) throws IOException, Globber.BadGlobException, InterruptedException {
        for (String string : includes) {
            Future<List<Path>> future = this.getGlobUnsortedAsync(string, globberOperation);
        }
        HashSet<String> results = new HashSet<String>();
        for (String pattern : includes) {
            List<String> items = this.getGlobUnsorted(pattern, globberOperation);
            if (!allowEmpty && items.isEmpty()) {
                GlobberUtils.throwBadGlobExceptionEmptyResult(pattern, globberOperation);
            }
            results.addAll(items);
        }
        try {
            UnixGlob.removeExcludes(results, excludes);
        }
        catch (UnixGlob.BadPattern badPattern) {
            throw new Globber.BadGlobException(badPattern.getMessage());
        }
        if (!allowEmpty && results.isEmpty()) {
            GlobberUtils.throwBadGlobExceptionAllExcluded(globberOperation);
        }
        return new ArrayList<String>(results);
    }

    public Set<Pair<String, Globber.Operation>> getKeySet() {
        return this.globCache.keySet();
    }

    public void finishBackgroundTasks() {
        GlobCache.finishBackgroundTasks(this.globCache.values());
    }

    private static void finishBackgroundTasks(Collection<Future<List<Path>>> tasks) {
        for (Future<List<Path>> task : tasks) {
            try {
                GlobCache.fromFuture(task);
            }
            catch (IOException | InterruptedException | CancellationException exception) {}
        }
    }

    public void cancelBackgroundTasks() {
        GlobCache.cancelBackgroundTasks(this.globCache.values());
    }

    private static void cancelBackgroundTasks(Collection<Future<List<Path>>> tasks) {
        for (Future<List<Path>> task : tasks) {
            task.cancel(true);
        }
        for (Future<List<Path>> task : tasks) {
            try {
                task.get();
            }
            catch (InterruptedException | CancellationException | ExecutionException exception) {}
        }
    }

    public String toString() {
        return "GlobCache for " + this.packageId + " in " + this.packageDirectory;
    }

    private class GlobUnixPathDiscriminator
    implements UnixGlobPathDiscriminator {
        private final Globber.Operation globberOperation;

        GlobUnixPathDiscriminator(Globber.Operation globberOperation) {
            this.globberOperation = globberOperation;
        }

        @Override
        public boolean shouldTraverseDirectory(Path directory) {
            return GlobCache.this.globCacheShouldTraverseDirectory(directory);
        }

        @Override
        public boolean shouldIncludePathInResult(Path path, boolean isDirectory) {
            switch (this.globberOperation) {
                case FILES_AND_DIRS: {
                    return !isDirectory || !GlobCache.this.isSubPackage(path);
                }
                case SUBPACKAGES: {
                    if (!isDirectory || path.equals(GlobCache.this.packageDirectory)) {
                        return false;
                    }
                    return GlobCache.this.isSubPackage(path);
                }
                case FILES: {
                    return !isDirectory;
                }
            }
            throw new IllegalStateException("Unexpected unhandled Globber.Operation enum value: " + this.globberOperation);
        }
    }
}

