/*
 * Decompiled with CFR 0.152.
 */
package com.google.devtools.build.lib.concurrent;

import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.MoreObjects;
import com.google.common.base.Preconditions;
import com.google.common.flogger.GoogleLogger;
import com.google.devtools.build.lib.concurrent.AddressFreer;
import com.google.devtools.build.lib.concurrent.ComparableRunnable;
import com.google.devtools.build.lib.concurrent.ErrorClassifier;
import com.google.devtools.build.lib.concurrent.PaddedAddresses;
import com.google.devtools.build.lib.concurrent.TaskFifo;
import com.google.devtools.build.lib.unsafe.UnsafeProvider;
import java.lang.ref.Cleaner;
import java.lang.ref.PhantomReference;
import java.lang.ref.ReferenceQueue;
import java.util.TreeMap;
import java.util.concurrent.ConcurrentSkipListSet;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.ForkJoinWorkerThread;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import javax.annotation.Nullable;
import sun.misc.Unsafe;

final class PriorityWorkerPool {
    private static final GoogleLogger logger = GoogleLogger.forEnclosingClass();
    private final int poolSize;
    private final int cpuPermits;
    private ForkJoinPool pool;
    private final TaskFifo queue;
    private final ConcurrentSkipListSet<ComparableRunnable> cpuHeavyQueue = new ConcurrentSkipListSet();
    private final String name;
    private final Cache<WorkerThread, Object> workers = Caffeine.newBuilder().weakKeys().build();
    private final Object quiescenceMonitor = new Object();
    private final ErrorClassifier errorClassifier;
    private final AtomicReference<Throwable> unhandled = new AtomicReference();
    private static final Runnable RUN_TASK = new LoopStarter(NextWorkerActivity.DO_TASK);
    private static final Runnable RUN_CPU_HEAVY_TASK = new LoopStarter(NextWorkerActivity.DO_CPU_HEAVY_TASK);
    private static final long CANCEL_BIT = Long.MIN_VALUE;
    private static final long CPU_PERMITS_MASK = 0x7FF0000000000000L;
    private static final int CPU_PERMITS_BIT_OFFSET = 52;
    private static final long ONE_CPU_PERMIT = 0x10000000000000L;
    private static final long THREADS_MASK = 0xFFFE000000000L;
    private static final int THREADS_BIT_OFFSET = 37;
    private static final long ONE_THREAD = 0x2000000000L;
    private static final long TASKS_MASK = 137430564864L;
    private static final int TASKS_BIT_OFFSET = 23;
    private static final long ONE_TASK = 0x800000L;
    static final int TASKS_MAX_VALUE = 16383;
    private static final long CPU_HEAVY_TASKS_MASK = 0x7FFFFFL;
    private static final int CPU_HEAVY_TASKS_BIT_OFFSET = 0;
    private static final long ONE_CPU_HEAVY_TASK = 1L;
    private static final long CPU_HEAVY_RESOURCES = 0x10002000000000L;
    private final long countersAddress;
    private static final Unsafe UNSAFE;

    PriorityWorkerPool(Cleaner cleaner, String name, int poolSize, int cpuPermits, ErrorClassifier errorClassifier) {
        Preconditions.checkArgument((long)poolSize <= 32767L, poolSize);
        Preconditions.checkArgument((long)cpuPermits <= 2047L, cpuPermits);
        this.name = name;
        this.poolSize = poolSize;
        this.cpuPermits = cpuPermits;
        this.pool = this.newForkJoinPool();
        this.errorClassifier = errorClassifier;
        long baseAddress = PaddedAddresses.createPaddedBaseAddress(4);
        cleaner.register(this, new AddressFreer(baseAddress));
        this.countersAddress = PaddedAddresses.getAlignedAddress(baseAddress, 0);
        this.queue = new TaskFifo(PaddedAddresses.getAlignedAddress(baseAddress, 1), PaddedAddresses.getAlignedAddress(baseAddress, 2), PaddedAddresses.getAlignedAddress(baseAddress, 3));
        this.resetExecutionCounters();
    }

    int poolSize() {
        return this.poolSize;
    }

    int cpuPermits() {
        return this.cpuPermits;
    }

    void execute(Runnable rawTask) {
        ComparableRunnable task;
        if (rawTask instanceof ComparableRunnable && (task = (ComparableRunnable)rawTask).isCpuHeavy()) {
            this.cpuHeavyQueue.add(task);
            if (this.acquireThreadAndCpuPermitElseReleaseCpuHeavyTask()) {
                this.pool.execute(RUN_CPU_HEAVY_TASK);
            }
            return;
        }
        while (!this.queue.tryAppend(rawTask)) {
            if (!this.tryAcquireTask()) {
                if (this.isCancelled()) {
                    return;
                }
                ((GoogleLogger.Api)((GoogleLogger.Api)logger.atWarning()).atMostEvery(5, TimeUnit.SECONDS)).log("Queue is full but no tasks could be acquired: %s", this);
                continue;
            }
            this.dequeueTaskAndRun();
        }
        if (this.acquireThreadElseReleaseTask()) {
            this.pool.execute(RUN_TASK);
        }
    }

    Object quiescenceMonitor() {
        return this.quiescenceMonitor;
    }

    @Nullable
    Throwable unhandled() {
        return this.unhandled.get();
    }

    void cancel() {
        this.markCancelled();
        this.workers.asMap().keySet().forEach(Thread::interrupt);
    }

    private void cleanup() {
        Preconditions.checkState(this.isQuiescent(), "cleanup called on pool that was not quiescent: %s", (Object)this);
        this.queue.clear();
        this.cpuHeavyQueue.clear();
        this.workers.invalidateAll();
        this.pool.shutdown();
    }

    void reset() {
        this.cleanup();
        this.unhandled.set(null);
        this.resetExecutionCounters();
        this.pool = this.newForkJoinPool();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void dispose() {
        this.cancel();
        Object object = this.quiescenceMonitor;
        synchronized (object) {
            while (!this.isQuiescent()) {
                try {
                    this.quiescenceMonitor.wait();
                }
                catch (InterruptedException e) {
                    ((GoogleLogger.Api)((GoogleLogger.Api)logger.atWarning()).withCause(e)).log("%s interrupted while cleaning up.", this);
                }
            }
        }
        this.cleanup();
    }

    @VisibleForTesting
    PhantomReference<ForkJoinPool> registerPoolDisposalMonitorForTesting(ReferenceQueue<ForkJoinPool> referenceQueue) {
        return new PhantomReference<ForkJoinPool>(this.pool, referenceQueue);
    }

    public String toString() {
        TreeMap<Thread.State, Integer> threadStates = new TreeMap<Thread.State, Integer>();
        for (WorkerThread w : this.workers.asMap().keySet()) {
            threadStates.compute(w.getState(), (k, v) -> v == null ? 1 : v + 1);
        }
        return MoreObjects.toStringHelper(this).add("available", PriorityWorkerPool.formatSnapshot(this.getExecutionCounters())).add("|queue|", this.queue.size()).add("|cpu queue|", this.cpuHeavyQueue.size()).add("threads", threadStates).add("unhandled", this.unhandled.get()).add("pool", this.pool).toString();
    }

    private void handleUncaughtError(Throwable error) {
        Throwable unhandledSnapshot;
        boolean critical = false;
        ErrorClassifier.ErrorClassification classification = this.errorClassifier.classify(error);
        switch (classification) {
            case AS_CRITICAL_AS_POSSIBLE: 
            case CRITICAL_AND_LOG: {
                ((GoogleLogger.Api)((GoogleLogger.Api)logger.atWarning()).withCause(error)).log("Found critical error in queue visitor");
            }
            case CRITICAL: {
                critical = true;
                break;
            }
        }
        while (!((unhandledSnapshot = this.unhandled.get()) != null && this.errorClassifier.classify(unhandledSnapshot).compareTo(classification) >= 0 || this.unhandled.compareAndSet(unhandledSnapshot, error))) {
        }
        if (critical) {
            this.cancel();
        }
    }

    private ForkJoinPool newForkJoinPool() {
        return new ForkJoinPool(this.poolSize, pool -> {
            WorkerThread worker = new WorkerThread(pool, this.name);
            this.workers.put(worker, "A non-null value, as required by Caffeine.");
            return worker;
        }, null, false);
    }

    private void dequeueTaskAndRun() {
        try {
            Runnable task = this.queue.take();
            task.run();
        }
        catch (Throwable uncaught) {
            this.handleUncaughtError(uncaught);
        }
    }

    private void dequeueCpuHeavyTaskAndRun() {
        try {
            this.cpuHeavyQueue.pollFirst().run();
        }
        catch (Throwable uncaught) {
            this.handleUncaughtError(uncaught);
        }
    }

    boolean isQuiescent() {
        long snapshot = this.getExecutionCounters();
        int threadsSnapshot = (int)((snapshot & 0xFFFE000000000L) >> 37);
        return threadsSnapshot == this.poolSize;
    }

    boolean isCancelled() {
        return this.getExecutionCounters() < 0L;
    }

    private void markCancelled() {
        long snapshot;
        do {
            if ((snapshot = this.getExecutionCounters()) >= 0L) continue;
            return;
        } while (!this.tryUpdateExecutionCounters(snapshot, snapshot | Long.MIN_VALUE));
    }

    private void resetExecutionCounters() {
        UNSAFE.putLong(null, this.countersAddress, (long)this.poolSize << 37 | (long)this.cpuPermits << 52);
    }

    private boolean acquireThreadElseReleaseTask() {
        boolean acquired;
        long target;
        long snapshot;
        while (!UNSAFE.compareAndSwapLong(null, this.countersAddress, snapshot, target = snapshot + ((acquired = ((snapshot = UNSAFE.getLongVolatile(null, this.countersAddress)) & 0xFFFE000000000L) > 0L && snapshot >= 0L) ? -137438953472L : 0x800000L))) {
        }
        return acquired;
    }

    private boolean acquireThreadAndCpuPermitElseReleaseCpuHeavyTask() {
        boolean acquired;
        long target;
        long snapshot;
        while (!UNSAFE.compareAndSwapLong(null, this.countersAddress, snapshot, target = snapshot + ((acquired = ((snapshot = UNSAFE.getLongVolatile(null, this.countersAddress)) & 0xFFF0000000000000L) > 0L && (snapshot & 0xFFFE000000000L) > 0L) ? -4503737066323968L : 1L))) {
        }
        return acquired;
    }

    private boolean tryAcquireTask() {
        long snapshot;
        do {
            if (((snapshot = this.getExecutionCounters()) & 0x1FFF800000L) != 0L && snapshot >= 0L) continue;
            return false;
        } while (!this.tryUpdateExecutionCounters(snapshot, snapshot - 0x800000L));
        return true;
    }

    private NextWorkerActivity getActivityFollowingTask() {
        long snapshot = UNSAFE.getLongVolatile(null, this.countersAddress);
        while (true) {
            if ((snapshot & 0x8000001FFF800000L) > 0L) {
                if (UNSAFE.compareAndSwapLong(null, this.countersAddress, snapshot, snapshot - 0x800000L)) {
                    return NextWorkerActivity.DO_TASK;
                }
            } else if ((snapshot & 0x80000000007FFFFFL) > 0L && (snapshot & 0x7FF0000000000000L) != 0L) {
                if (UNSAFE.compareAndSwapLong(null, this.countersAddress, snapshot, snapshot - 0x10000000000001L)) {
                    return NextWorkerActivity.DO_CPU_HEAVY_TASK;
                }
            } else {
                long target = snapshot + 0x2000000000L;
                if (UNSAFE.compareAndSwapLong(null, this.countersAddress, snapshot, target)) {
                    return this.quiescentOrIdle(target);
                }
            }
            snapshot = UNSAFE.getLong(null, this.countersAddress);
        }
    }

    private NextWorkerActivity getActivityFollowingCpuHeavyTask() {
        long snapshot = UNSAFE.getLongVolatile(null, this.countersAddress);
        while (true) {
            if ((snapshot & 0x8000001FFF800000L) > 0L) {
                if (UNSAFE.compareAndSwapLong(null, this.countersAddress, snapshot, snapshot + 0xFFFFFFF800000L)) {
                    return NextWorkerActivity.DO_TASK;
                }
            } else if ((snapshot & 0x80000000007FFFFFL) > 0L) {
                if (UNSAFE.compareAndSwapLong(null, this.countersAddress, snapshot, snapshot - 1L)) {
                    return NextWorkerActivity.DO_CPU_HEAVY_TASK;
                }
            } else {
                long target = snapshot + 0x10002000000000L;
                if (UNSAFE.compareAndSwapLong(null, this.countersAddress, snapshot, target)) {
                    return this.quiescentOrIdle(target);
                }
            }
            snapshot = UNSAFE.getLong(null, this.countersAddress);
        }
    }

    private NextWorkerActivity quiescentOrIdle(long snapshot) {
        int snapshotThreads = (int)((snapshot & 0xFFFE000000000L) >> 37);
        return snapshotThreads == this.poolSize ? NextWorkerActivity.QUIESCENT : NextWorkerActivity.IDLE;
    }

    private long getExecutionCounters() {
        return UNSAFE.getLongVolatile(null, this.countersAddress);
    }

    private boolean tryUpdateExecutionCounters(long snapshot, long target) {
        return UNSAFE.compareAndSwapLong(null, this.countersAddress, snapshot, target);
    }

    private static String formatSnapshot(long snapshot) {
        return String.format("{cancelled=%b, threads=%d, cpuPermits=%d, tasks=%d, cpuHeavyTasks=%d}", snapshot < 0L, (snapshot & 0xFFFE000000000L) >> 37, (snapshot & 0x7FF0000000000000L) >> 52, (snapshot & 0x1FFF800000L) >> 23, (snapshot & 0x7FFFFFL) >> 0);
    }

    static {
        Preconditions.checkState(true, "Inconsistent CPU Permits Constants");
        Preconditions.checkState(true, "Inconistent Threads Constants");
        Preconditions.checkState(true, "Inconsistent CPU Heavy Task Constants");
        UNSAFE = UnsafeProvider.unsafe();
    }

    class WorkerThread
    extends ForkJoinWorkerThread {
        private WorkerThread(ForkJoinPool pool, String name) {
            super(pool);
            this.setName(name + "-" + this.getPoolIndex());
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void runLoop(NextWorkerActivity nextActivity) {
            while (true) {
                switch (nextActivity) {
                    case QUIESCENT: {
                        Object object = PriorityWorkerPool.this.quiescenceMonitor;
                        synchronized (object) {
                            PriorityWorkerPool.this.quiescenceMonitor.notifyAll();
                        }
                        return;
                    }
                    case IDLE: {
                        return;
                    }
                    case DO_TASK: {
                        PriorityWorkerPool.this.dequeueTaskAndRun();
                        nextActivity = PriorityWorkerPool.this.getActivityFollowingTask();
                        break;
                    }
                    case DO_CPU_HEAVY_TASK: {
                        PriorityWorkerPool.this.dequeueCpuHeavyTaskAndRun();
                        nextActivity = PriorityWorkerPool.this.getActivityFollowingCpuHeavyTask();
                    }
                }
            }
        }

        boolean tryDoQueuedWork() {
            if (!PriorityWorkerPool.this.tryAcquireTask()) {
                return false;
            }
            PriorityWorkerPool.this.dequeueTaskAndRun();
            return true;
        }
    }

    private static class LoopStarter
    implements Runnable {
        private final NextWorkerActivity activity;

        private LoopStarter(NextWorkerActivity activity) {
            this.activity = activity;
        }

        @Override
        public void run() {
            ((WorkerThread)Thread.currentThread()).runLoop(this.activity);
        }
    }

    static enum NextWorkerActivity {
        QUIESCENT,
        IDLE,
        DO_TASK,
        DO_CPU_HEAVY_TASK;

    }
}

