/*
 * Decompiled with CFR 0.152.
 */
package net.starlark.java.eval;

import com.google.common.base.Ascii;
import com.google.common.base.CharMatcher;
import com.google.common.base.Joiner;
import java.util.ArrayList;
import java.util.Collections;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import net.starlark.java.annot.Param;
import net.starlark.java.annot.ParamType;
import net.starlark.java.annot.StarlarkBuiltin;
import net.starlark.java.annot.StarlarkMethod;
import net.starlark.java.eval.Dict;
import net.starlark.java.eval.EvalException;
import net.starlark.java.eval.EvalUtils;
import net.starlark.java.eval.FormatParser;
import net.starlark.java.eval.NoneType;
import net.starlark.java.eval.RangeList;
import net.starlark.java.eval.Sequence;
import net.starlark.java.eval.Starlark;
import net.starlark.java.eval.StarlarkInt;
import net.starlark.java.eval.StarlarkList;
import net.starlark.java.eval.StarlarkThread;
import net.starlark.java.eval.StarlarkValue;
import net.starlark.java.eval.Tuple;

@StarlarkBuiltin(name="string", category="core", doc="A language built-in type to support strings. Examples of string literals:<br><pre class=\"language-python\">a = 'abc\\ndef'\nb = \"ab'cd\"\nc = \"\"\"multiline string\"\"\"\n\n# Strings support slicing (negative index starts from the end):\nx = \"hello\"[2:4]  # \"ll\"\ny = \"hello\"[1:-1]  # \"ell\"\nz = \"hello\"[:4]  # \"hell\"# Slice steps can be used, too:\ns = \"hello\"[::2] # \"hlo\"\nt = \"hello\"[3:0:-1] # \"lle\"\n</pre>Strings are not directly iterable, use the <code>.elems()</code> method to iterate over their characters. Examples:<br><pre class=\"language-python\">\"bc\" in \"abcd\"   # evaluates to True\nx = [c for c in \"abc\".elems()]  # x == [\"a\", \"b\", \"c\"]</pre>\nImplicit concatenation of strings is not allowed; use the <code>+</code> operator instead. Comparison operators perform a lexicographical comparison; use <code>==</code> to test for equality.")
final class StringModule
implements StarlarkValue {
    static final StringModule INSTANCE = new StringModule();
    private static final String[] ASCII_CHAR_STRINGS = StringModule.initCharStrings();
    private static final String LATIN1_WHITESPACE = "\t\n\u000b\f\r\u001c\u001d\u001e\u001f \u0085\u00a0";
    private static final Pattern SPLIT_LINES_PATTERN = Pattern.compile("(?<line>.*)(?<break>(\\r\\n|\\r|\\n)?)");
    private static final CharMatcher DIGIT = CharMatcher.javaDigit();
    private static final CharMatcher LOWER = CharMatcher.inRange('a', 'z');
    private static final CharMatcher UPPER = CharMatcher.inRange('A', 'Z');
    private static final CharMatcher ALPHA = LOWER.or(UPPER);
    private static final CharMatcher ALNUM = ALPHA.or(DIGIT);
    private static final CharMatcher CASED = ALPHA;
    private static final CharMatcher SPACE = CharMatcher.whitespace();

    private StringModule() {
    }

    static String slice(String s2, int start, int stop, int step) throws EvalException {
        RangeList indices = new RangeList(start, stop, step);
        int n = indices.size();
        if (n == 0) {
            return "";
        }
        if (n == 1) {
            return StringModule.memoizedCharToString(s2.charAt(indices.at(0)));
        }
        if (step == 1) {
            return s2.substring(indices.at(0), indices.at(n));
        }
        char[] res = new char[n];
        for (int i = 0; i < n; ++i) {
            res[i] = s2.charAt(indices.at(i));
        }
        return new String(res);
    }

    private static String[] initCharStrings() {
        String[] a = new String[128];
        for (int i = 0; i < a.length; ++i) {
            a[i] = String.valueOf((char)i);
        }
        return a;
    }

    static String memoizedCharToString(char c) {
        if (c < ASCII_CHAR_STRINGS.length) {
            return ASCII_CHAR_STRINGS[c];
        }
        return String.valueOf(c);
    }

    private static long substringIndices(String str, Object start, Object end) throws EvalException {
        int n = str.length();
        int istart = 0;
        if (start != Starlark.NONE) {
            istart = EvalUtils.toIndex(Starlark.toInt(start, "start"), n);
        }
        int iend = n;
        if (end != Starlark.NONE) {
            iend = EvalUtils.toIndex(Starlark.toInt(end, "end"), n);
        }
        if (iend < istart) {
            iend = istart;
        }
        return StringModule.pack(istart, iend);
    }

    private static long pack(int lo, int hi) {
        return (long)hi << 32 | (long)lo & 0xFFFFFFFFL;
    }

    private static int lo(long x) {
        return (int)x;
    }

    private static int hi(long x) {
        return (int)(x >>> 32);
    }

    @StarlarkMethod(name="join", doc="Returns a string in which the string elements of the argument have been joined by this string as a separator. Example:<br><pre class=\"language-python\">\"|\".join([\"a\", \"b\", \"c\"]) == \"a|b|c\"</pre>", parameters={@Param(name="self"), @Param(name="elements", doc="The objects to join.")}, useStarlarkThread=true)
    public String join(String self, Object elements, StarlarkThread thread) throws EvalException {
        Iterable<?> items = Starlark.toIterable(elements);
        int i = 0;
        for (Object item : items) {
            if (!(item instanceof String)) {
                throw Starlark.errorf("expected string for sequence element %d, got '%s' of type %s", i, Starlark.str(item, thread.getSemantics()), Starlark.type(item));
            }
            ++i;
        }
        return Joiner.on(self).join(items);
    }

    @StarlarkMethod(name="lower", doc="Returns the lower case version of this string.", parameters={@Param(name="self")})
    public String lower(String self) {
        return Ascii.toLowerCase(self);
    }

    @StarlarkMethod(name="upper", doc="Returns the upper case version of this string.", parameters={@Param(name="self")})
    public String upper(String self) {
        return Ascii.toUpperCase(self);
    }

    private static String stringLStrip(String self, String chars) {
        CharMatcher matcher = CharMatcher.anyOf(chars);
        for (int i = 0; i < self.length(); ++i) {
            if (matcher.matches(self.charAt(i))) continue;
            return self.substring(i);
        }
        return "";
    }

    private static String stringRStrip(String self, String chars) {
        CharMatcher matcher = CharMatcher.anyOf(chars);
        for (int i = self.length() - 1; i >= 0; --i) {
            if (matcher.matches(self.charAt(i))) continue;
            return self.substring(0, i + 1);
        }
        return "";
    }

    private static String stringStrip(String self, String chars) {
        return StringModule.stringLStrip(StringModule.stringRStrip(self, chars), chars);
    }

    @StarlarkMethod(name="lstrip", doc="Returns a copy of the string where leading characters that appear in <code>chars</code> are removed. Note that <code>chars</code> is not a prefix: all combinations of its value are removed:<pre class=\"language-python\">\"abcba\".lstrip(\"ba\") == \"cba\"</pre>", parameters={@Param(name="self"), @Param(name="chars", allowedTypes={@ParamType(type=String.class), @ParamType(type=NoneType.class)}, doc="The characters to remove, or all whitespace if None.", defaultValue="None")})
    public String lstrip(String self, Object charsOrNone) {
        String chars = charsOrNone != Starlark.NONE ? (String)charsOrNone : LATIN1_WHITESPACE;
        return StringModule.stringLStrip(self, chars);
    }

    @StarlarkMethod(name="rstrip", doc="Returns a copy of the string where trailing characters that appear in <code>chars</code> are removed. Note that <code>chars</code> is not a suffix: all combinations of its value are removed:<pre class=\"language-python\">\"abcbaa\".rstrip(\"ab\") == \"abc\"</pre>", parameters={@Param(name="self", doc="This string."), @Param(name="chars", allowedTypes={@ParamType(type=String.class), @ParamType(type=NoneType.class)}, doc="The characters to remove, or all whitespace if None.", defaultValue="None")})
    public String rstrip(String self, Object charsOrNone) {
        String chars = charsOrNone != Starlark.NONE ? (String)charsOrNone : LATIN1_WHITESPACE;
        return StringModule.stringRStrip(self, chars);
    }

    @StarlarkMethod(name="strip", doc="Returns a copy of the string where leading or trailing characters that appear in <code>chars</code> are removed. Note that <code>chars</code> is neither a prefix nor a suffix: all combinations of its value are removed:<pre class=\"language-python\">\"aabcbcbaa\".strip(\"ab\") == \"cbc\"</pre>", parameters={@Param(name="self", doc="This string."), @Param(name="chars", allowedTypes={@ParamType(type=String.class), @ParamType(type=NoneType.class)}, doc="The characters to remove, or all whitespace if None.", defaultValue="None")})
    public String strip(String self, Object charsOrNone) {
        String chars = charsOrNone != Starlark.NONE ? (String)charsOrNone : LATIN1_WHITESPACE;
        return StringModule.stringStrip(self, chars);
    }

    @StarlarkMethod(name="replace", doc="Returns a copy of the string in which the occurrences of <code>old</code> have been replaced with <code>new</code>, optionally restricting the number of replacements to <code>count</code>.", parameters={@Param(name="self", doc="This string."), @Param(name="old", doc="The string to be replaced."), @Param(name="new", doc="The string to replace with."), @Param(name="count", defaultValue="-1", doc="The maximum number of replacements. If omitted, or if the value is negative, there is no limit.")}, useStarlarkThread=true)
    public String replace(String self, String oldString, String newString, StarlarkInt countI, StarlarkThread thread) throws EvalException {
        int count = countI.toInt("count");
        if (count < 0) {
            count = Integer.MAX_VALUE;
        }
        StringBuilder sb = new StringBuilder();
        int start = 0;
        for (int i = 0; i < count; ++i) {
            if (oldString.isEmpty()) {
                sb.append(newString);
                if (start >= self.length()) break;
                sb.append(self.charAt(start++));
                continue;
            }
            int end = self.indexOf(oldString, start);
            if (end < 0) break;
            sb.append(self, start, end).append(newString);
            start = end + oldString.length();
        }
        sb.append(self, start, self.length());
        return sb.toString();
    }

    @StarlarkMethod(name="split", doc="Returns a list of all the words in the string, using <code>sep</code> as the separator, optionally limiting the number of splits to <code>maxsplit</code>.", parameters={@Param(name="self", doc="This string."), @Param(name="sep", doc="The string to split on."), @Param(name="maxsplit", allowedTypes={@ParamType(type=StarlarkInt.class), @ParamType(type=NoneType.class)}, defaultValue="None", doc="The maximum number of splits.")}, useStarlarkThread=true)
    public StarlarkList<String> split(String self, String sep, Object maxSplitO, StarlarkThread thread) throws EvalException {
        if (sep.isEmpty()) {
            throw Starlark.errorf("Empty separator", new Object[0]);
        }
        int maxSplit = Integer.MAX_VALUE;
        if (maxSplitO != Starlark.NONE) {
            maxSplit = Starlark.toInt(maxSplitO, "maxsplit");
        }
        StarlarkList<String> res = StarlarkList.newList(thread.mutability());
        int start = 0;
        while (true) {
            int end;
            if ((end = self.indexOf(sep, start)) < 0 || maxSplit-- == 0) break;
            res.addElement(self.substring(start, end));
            start = end + sep.length();
        }
        res.addElement(self.substring(start));
        return res;
    }

    @StarlarkMethod(name="rsplit", doc="Returns a list of all the words in the string, using <code>sep</code> as the separator, optionally limiting the number of splits to <code>maxsplit</code>. Except for splitting from the right, this method behaves like split().", parameters={@Param(name="self", doc="This string."), @Param(name="sep", doc="The string to split on."), @Param(name="maxsplit", allowedTypes={@ParamType(type=StarlarkInt.class), @ParamType(type=NoneType.class)}, defaultValue="None", doc="The maximum number of splits.")}, useStarlarkThread=true)
    public StarlarkList<String> rsplit(String self, String sep, Object maxSplitO, StarlarkThread thread) throws EvalException {
        if (sep.isEmpty()) {
            throw Starlark.errorf("Empty separator", new Object[0]);
        }
        int maxSplit = Integer.MAX_VALUE;
        if (maxSplitO != Starlark.NONE) {
            maxSplit = Starlark.toInt(maxSplitO, "maxsplit");
        }
        ArrayList<String> res = new ArrayList<String>();
        int end = self.length();
        while (true) {
            int start;
            if ((start = self.lastIndexOf(sep, end - 1)) < 0 || maxSplit-- == 0) break;
            res.add(self.substring(start + sep.length(), end));
            end = start;
        }
        res.add(self.substring(0, end));
        Collections.reverse(res);
        return StarlarkList.copyOf(thread.mutability(), res);
    }

    @StarlarkMethod(name="partition", doc="Splits the input string at the first occurrence of the separator <code>sep</code> and returns the resulting partition as a three-element tuple of the form (before, separator, after). If the input string does not contain the separator, partition returns (self, '', '').", parameters={@Param(name="self"), @Param(name="sep", doc="The string to split on.")})
    public Tuple partition(String self, String sep) throws EvalException {
        return StringModule.partitionCommon(self, sep, true);
    }

    @StarlarkMethod(name="rpartition", doc="Splits the input string at the last occurrence of the separator <code>sep</code> and returns the resulting partition as a three-element tuple of the form (before, separator, after). If the input string does not contain the separator, rpartition returns ('', '', self).", parameters={@Param(name="self"), @Param(name="sep", doc="The string to split on.")})
    public Tuple rpartition(String self, String sep) throws EvalException {
        return StringModule.partitionCommon(self, sep, false);
    }

    private static Tuple partitionCommon(String input, String separator, boolean first) throws EvalException {
        int pos;
        if (separator.isEmpty()) {
            throw Starlark.errorf("empty separator", new Object[0]);
        }
        String a = "";
        String b = "";
        String c = "";
        int n = pos = first ? input.indexOf(separator) : input.lastIndexOf(separator);
        if (pos < 0) {
            if (first) {
                a = input;
            } else {
                c = input;
            }
        } else {
            a = input.substring(0, pos);
            b = separator;
            c = input.substring(pos + separator.length());
        }
        return Tuple.triple(a, b, c);
    }

    @StarlarkMethod(name="capitalize", doc="Returns a copy of the string with its first character (if any) capitalized and the rest lowercased. This method does not support non-ascii characters. ", parameters={@Param(name="self", doc="This string.")})
    public String capitalize(String self) throws EvalException {
        if (self.isEmpty()) {
            return self;
        }
        return Character.toUpperCase(self.charAt(0)) + Ascii.toLowerCase(self.substring(1));
    }

    @StarlarkMethod(name="title", doc="Converts the input string into title case, i.e. every word starts with an uppercase letter while the remaining letters are lowercase. In this context, a word means strictly a sequence of letters. This method does not support supplementary Unicode characters.", parameters={@Param(name="self", doc="This string.")})
    public String title(String self) throws EvalException {
        char[] data = self.toCharArray();
        boolean previousWasLetter = false;
        for (int pos = 0; pos < data.length; ++pos) {
            char current = data[pos];
            boolean currentIsLetter = Character.isLetter(current);
            if (currentIsLetter) {
                if (previousWasLetter && Character.isUpperCase(current)) {
                    data[pos] = Character.toLowerCase(current);
                } else if (!previousWasLetter && Character.isLowerCase(current)) {
                    data[pos] = Character.toUpperCase(current);
                }
            }
            previousWasLetter = currentIsLetter;
        }
        return new String(data);
    }

    private static int stringFind(boolean forward, String self, String sub, Object start, Object end) throws EvalException {
        long indices = StringModule.substringIndices(self, start, end);
        int startpos = StringModule.lo(indices);
        int endpos = StringModule.hi(indices);
        if (forward && endpos == self.length()) {
            return self.indexOf(sub, startpos);
        }
        String substr = self.substring(startpos, endpos);
        int subpos = forward ? substr.indexOf(sub) : substr.lastIndexOf(sub);
        return subpos < 0 ? subpos : subpos + startpos;
    }

    @StarlarkMethod(name="rfind", doc="Returns the last index where <code>sub</code> is found, or -1 if no such index exists, optionally restricting to <code>[start:end]</code>, <code>start</code> being inclusive and <code>end</code> being exclusive.", parameters={@Param(name="self", doc="This string."), @Param(name="sub", doc="The substring to find."), @Param(name="start", allowedTypes={@ParamType(type=StarlarkInt.class), @ParamType(type=NoneType.class)}, defaultValue="0", doc="Restrict to search from this position."), @Param(name="end", allowedTypes={@ParamType(type=StarlarkInt.class), @ParamType(type=NoneType.class)}, defaultValue="None", doc="optional position before which to restrict to search.")})
    public int rfind(String self, String sub, Object start, Object end) throws EvalException {
        return StringModule.stringFind(false, self, sub, start, end);
    }

    @StarlarkMethod(name="find", doc="Returns the first index where <code>sub</code> is found, or -1 if no such index exists, optionally restricting to <code>[start:end]</code>, <code>start</code> being inclusive and <code>end</code> being exclusive.", parameters={@Param(name="self", doc="This string."), @Param(name="sub", doc="The substring to find."), @Param(name="start", allowedTypes={@ParamType(type=StarlarkInt.class), @ParamType(type=NoneType.class)}, defaultValue="0", doc="Restrict to search from this position."), @Param(name="end", allowedTypes={@ParamType(type=StarlarkInt.class), @ParamType(type=NoneType.class)}, defaultValue="None", doc="optional position before which to restrict to search.")})
    public int find(String self, String sub, Object start, Object end) throws EvalException {
        return StringModule.stringFind(true, self, sub, start, end);
    }

    @StarlarkMethod(name="rindex", doc="Returns the last index where <code>sub</code> is found, or raises an error if no such index exists, optionally restricting to <code>[start:end]</code>, <code>start</code> being inclusive and <code>end</code> being exclusive.", parameters={@Param(name="self", doc="This string."), @Param(name="sub", doc="The substring to find."), @Param(name="start", allowedTypes={@ParamType(type=StarlarkInt.class), @ParamType(type=NoneType.class)}, defaultValue="0", doc="Restrict to search from this position."), @Param(name="end", allowedTypes={@ParamType(type=StarlarkInt.class), @ParamType(type=NoneType.class)}, defaultValue="None", doc="optional position before which to restrict to search.")})
    public int rindex(String self, String sub, Object start, Object end) throws EvalException {
        int res = StringModule.stringFind(false, self, sub, start, end);
        if (res < 0) {
            throw Starlark.errorf("substring not found", new Object[0]);
        }
        return res;
    }

    @StarlarkMethod(name="index", doc="Returns the first index where <code>sub</code> is found, or raises an error if no such  index exists, optionally restricting to <code>[start:end]</code><code>start</code> being inclusive and <code>end</code> being exclusive.", parameters={@Param(name="self", doc="This string."), @Param(name="sub", doc="The substring to find."), @Param(name="start", allowedTypes={@ParamType(type=StarlarkInt.class), @ParamType(type=NoneType.class)}, defaultValue="0", doc="Restrict to search from this position."), @Param(name="end", allowedTypes={@ParamType(type=StarlarkInt.class), @ParamType(type=NoneType.class)}, defaultValue="None", doc="optional position before which to restrict to search.")})
    public int index(String self, String sub, Object start, Object end) throws EvalException {
        int res = StringModule.stringFind(true, self, sub, start, end);
        if (res < 0) {
            throw Starlark.errorf("substring not found", new Object[0]);
        }
        return res;
    }

    @StarlarkMethod(name="splitlines", doc="Splits the string at line boundaries ('\\n', '\\r\\n', '\\r') and returns the result as a new mutable list.", parameters={@Param(name="self", doc="This string."), @Param(name="keepends", defaultValue="False", doc="Whether the line breaks should be included in the resulting list.")}, useStarlarkThread=true)
    public Sequence<String> splitLines(String self, boolean keepEnds, StarlarkThread thread) throws EvalException {
        StarlarkList<String> result = StarlarkList.newList(thread.mutability());
        Matcher matcher = SPLIT_LINES_PATTERN.matcher(self);
        while (matcher.find()) {
            String line = matcher.group("line");
            String lineBreak = matcher.group("break");
            boolean trailingBreak = lineBreak.isEmpty();
            if (line.isEmpty() && trailingBreak) break;
            if (keepEnds && !trailingBreak) {
                result.addElement(line + lineBreak);
                continue;
            }
            result.addElement(line);
        }
        return result;
    }

    @StarlarkMethod(name="isalpha", doc="Returns True if all characters in the string are alphabetic ([a-zA-Z]) and there is at least one character.", parameters={@Param(name="self", doc="This string.")})
    public boolean isAlpha(String self) throws EvalException {
        return StringModule.matches(self, ALPHA, false);
    }

    @StarlarkMethod(name="isalnum", doc="Returns True if all characters in the string are alphanumeric ([a-zA-Z0-9]) and there is at least one character.", parameters={@Param(name="self", doc="This string.")})
    public boolean isAlnum(String self) throws EvalException {
        return StringModule.matches(self, ALNUM, false);
    }

    @StarlarkMethod(name="isdigit", doc="Returns True if all characters in the string are digits ([0-9]) and there is at least one character.", parameters={@Param(name="self", doc="This string.")})
    public boolean isDigit(String self) throws EvalException {
        return StringModule.matches(self, DIGIT, false);
    }

    @StarlarkMethod(name="isspace", doc="Returns True if all characters are white space characters and the string contains at least one character.", parameters={@Param(name="self", doc="This string.")})
    public boolean isSpace(String self) throws EvalException {
        return StringModule.matches(self, SPACE, false);
    }

    @StarlarkMethod(name="islower", doc="Returns True if all cased characters in the string are lowercase and there is at least one character.", parameters={@Param(name="self", doc="This string.")})
    public boolean isLower(String self) throws EvalException {
        return StringModule.matches(self, UPPER.negate(), true);
    }

    @StarlarkMethod(name="isupper", doc="Returns True if all cased characters in the string are uppercase and there is at least one character.", parameters={@Param(name="self", doc="This string.")})
    public boolean isUpper(String self) throws EvalException {
        return StringModule.matches(self, LOWER.negate(), true);
    }

    @StarlarkMethod(name="istitle", doc="Returns True if the string is in title case and it contains at least one character. This means that every uppercase character must follow an uncased one (e.g. whitespace) and every lowercase character must follow a cased one (e.g. uppercase or lowercase).", parameters={@Param(name="self", doc="This string.")})
    public boolean isTitle(String self) throws EvalException {
        if (self.isEmpty()) {
            return false;
        }
        char[] data = self.toCharArray();
        CharMatcher matcher = CharMatcher.any();
        char leftMostCased = ' ';
        for (int pos = data.length - 1; pos >= 0; --pos) {
            char current = data[pos];
            if (!matcher.matches(current)) {
                return false;
            }
            matcher = LOWER.matches(current) ? CASED : (UPPER.matches(current) ? CASED.negate() : CharMatcher.any());
            if (!CASED.matches(current)) continue;
            leftMostCased = current;
        }
        return UPPER.matches(leftMostCased);
    }

    private static boolean matches(String str, CharMatcher matcher, boolean requiresAtLeastOneCasedLetter) {
        if (str.isEmpty()) {
            return false;
        }
        if (!requiresAtLeastOneCasedLetter) {
            return matcher.matchesAllOf(str);
        }
        int casedLetters = 0;
        for (char current : str.toCharArray()) {
            if (!matcher.matches(current)) {
                return false;
            }
            if (!requiresAtLeastOneCasedLetter || !CASED.matches(current)) continue;
            ++casedLetters;
        }
        return casedLetters > 0;
    }

    @StarlarkMethod(name="count", doc="Returns the number of (non-overlapping) occurrences of substring <code>sub</code> in string, optionally restricting to <code>[start:end]</code>, <code>start</code> being inclusive and <code>end</code> being exclusive.", parameters={@Param(name="self", doc="This string."), @Param(name="sub", doc="The substring to count."), @Param(name="start", allowedTypes={@ParamType(type=StarlarkInt.class), @ParamType(type=NoneType.class)}, defaultValue="0", doc="Restrict to search from this position."), @Param(name="end", allowedTypes={@ParamType(type=StarlarkInt.class), @ParamType(type=NoneType.class)}, defaultValue="None", doc="optional position before which to restrict to search.")})
    public int count(String self, String sub, Object start, Object end) throws EvalException {
        long indices = StringModule.substringIndices(self, start, end);
        if (sub.isEmpty()) {
            return StringModule.hi(indices) - StringModule.lo(indices) + 1;
        }
        String str = self.substring(StringModule.lo(indices), StringModule.hi(indices));
        int count = 0;
        int index = 0;
        while ((index = str.indexOf(sub, index)) >= 0) {
            ++count;
            index += sub.length();
        }
        return count;
    }

    @StarlarkMethod(name="elems", doc="Returns an iterable value containing successive 1-element substrings of the string. Equivalent to <code>[s[i] for i in range(len(s))]</code>, except that the returned value might not be a list.", parameters={@Param(name="self", doc="This string.")})
    public Sequence<String> elems(String self) {
        char[] chars = self.toCharArray();
        Object[] strings = new Object[chars.length];
        for (int i = 0; i < chars.length; ++i) {
            strings[i] = StringModule.memoizedCharToString(chars[i]);
        }
        return StarlarkList.wrap(null, strings);
    }

    @StarlarkMethod(name="endswith", doc="Returns True if the string ends with <code>sub</code>, otherwise False, optionally restricting to <code>[start:end]</code>, <code>start</code> being inclusive and <code>end</code> being exclusive.", parameters={@Param(name="self", doc="This string."), @Param(name="sub", allowedTypes={@ParamType(type=String.class), @ParamType(type=Tuple.class, generic1=String.class)}, doc="The suffix (or tuple of alternative suffixes) to match."), @Param(name="start", allowedTypes={@ParamType(type=StarlarkInt.class), @ParamType(type=NoneType.class)}, defaultValue="0", doc="Test beginning at this position."), @Param(name="end", allowedTypes={@ParamType(type=StarlarkInt.class), @ParamType(type=NoneType.class)}, defaultValue="None", doc="optional position at which to stop comparing.")})
    public boolean endsWith(String self, Object sub, Object start, Object end) throws EvalException {
        long indices = StringModule.substringIndices(self, start, end);
        if (sub instanceof String) {
            return StringModule.substringEndsWith(self, StringModule.lo(indices), StringModule.hi(indices), (String)sub);
        }
        for (String s2 : Sequence.cast(sub, String.class, "sub")) {
            if (!StringModule.substringEndsWith(self, StringModule.lo(indices), StringModule.hi(indices), s2)) continue;
            return true;
        }
        return false;
    }

    private static boolean substringEndsWith(String str, int start, int end, String suffix) {
        int n = suffix.length();
        return start + n <= end && str.regionMatches(end - n, suffix, 0, n);
    }

    @StarlarkMethod(name="format", doc="Perform string interpolation. Format strings contain replacement fields surrounded by curly braces <code>&#123;&#125;</code>. Anything that is not contained in braces is considered literal text, which is copied unchanged to the output.If you need to include a brace character in the literal text, it can be escaped by doubling: <code>&#123;&#123;</code> and <code>&#125;&#125;</code>A replacement field can be either a name, a number, or empty. Values are converted to strings using the <a href=\"../globals/all.html#str\">str</a> function.<pre class=\"language-python\"># Access in order:\n\"&#123;&#125; < &#123;&#125;\".format(4, 5) == \"4 < 5\"\n# Access by position:\n\"{1}, {0}\".format(2, 1) == \"1, 2\"\n# Access by name:\n\"x{key}x\".format(key = 2) == \"x2x\"</pre>\n", parameters={@Param(name="self", doc="This string.")}, extraPositionals=@Param(name="args", defaultValue="()", doc="List of arguments."), extraKeywords=@Param(name="kwargs", defaultValue="{}", doc="Dictionary of arguments."), useStarlarkThread=true)
    public String format(String self, Tuple args, Dict<String, Object> kwargs, StarlarkThread thread) throws EvalException {
        return new FormatParser().format(self, args, kwargs, thread.getSemantics());
    }

    @StarlarkMethod(name="startswith", doc="Returns True if the string starts with <code>sub</code>, otherwise False, optionally restricting to <code>[start:end]</code>, <code>start</code> being inclusive and <code>end</code> being exclusive.", parameters={@Param(name="self", doc="This string."), @Param(name="sub", allowedTypes={@ParamType(type=String.class), @ParamType(type=Tuple.class, generic1=String.class)}, doc="The prefix (or tuple of alternative prefixes) to match."), @Param(name="start", allowedTypes={@ParamType(type=StarlarkInt.class), @ParamType(type=NoneType.class)}, defaultValue="0", doc="Test beginning at this position."), @Param(name="end", allowedTypes={@ParamType(type=StarlarkInt.class), @ParamType(type=NoneType.class)}, defaultValue="None", doc="Stop comparing at this position.")})
    public boolean startsWith(String self, Object sub, Object start, Object end) throws EvalException {
        long indices = StringModule.substringIndices(self, start, end);
        if (sub instanceof String) {
            return StringModule.substringStartsWith(self, StringModule.lo(indices), StringModule.hi(indices), (String)sub);
        }
        for (String s2 : Sequence.cast(sub, String.class, "sub")) {
            if (!StringModule.substringStartsWith(self, StringModule.lo(indices), StringModule.hi(indices), s2)) continue;
            return true;
        }
        return false;
    }

    private static boolean substringStartsWith(String str, int start, int end, String prefix) {
        return start + prefix.length() <= end && str.startsWith(prefix, start);
    }

    @StarlarkMethod(name="removeprefix", doc="If the string starts with <code>prefix</code>, returns a new string with the prefix removed. Otherwise, returns the string.", parameters={@Param(name="self", doc="This string."), @Param(name="prefix", doc="The prefix to remove if present.")})
    public String removePrefix(String self, String prefix) {
        if (self.startsWith(prefix)) {
            return self.substring(prefix.length());
        }
        return self;
    }

    @StarlarkMethod(name="removesuffix", doc="If the string ends with <code>suffix</code>, returns a new string with the suffix removed. Otherwise, returns the string.", parameters={@Param(name="self", doc="This string."), @Param(name="suffix", doc="The suffix to remove if present.")})
    public String removeSuffix(String self, String suffix) {
        if (self.endsWith(suffix)) {
            return self.substring(0, self.length() - suffix.length());
        }
        return self;
    }
}

