/*
 * Decompiled with CFR 0.152.
 */
package net.starlark.java.eval;

import java.math.BigInteger;
import net.starlark.java.annot.StarlarkBuiltin;
import net.starlark.java.eval.EvalException;
import net.starlark.java.eval.Printer;
import net.starlark.java.eval.Starlark;
import net.starlark.java.eval.StarlarkFloat;
import net.starlark.java.eval.StarlarkValue;

@StarlarkBuiltin(name="int", category="core", doc="The type of integers in Starlark. Starlark integers may be of any magnitude; arithmetic is exact. Examples of integer expressions:<br><pre class=\"language-python\">153\n0x2A  # hexadecimal literal\n0o54  # octal literal\n23 * 2 + 5\n100 / -7\n100 % -7  # -5 (unlike in some other languages)\nint(\"18\")\n</pre>")
public abstract class StarlarkInt
implements StarlarkValue,
Comparable<StarlarkInt> {
    private static final int LEAST_SMALLINT = -128;
    private static final Int32[] smallints = new Int32[100000];
    static final StarlarkInt ZERO = StarlarkInt.of(0);
    private static final StarlarkInt ONE = StarlarkInt.of(1);
    private static final StarlarkInt MINUS_ONE = StarlarkInt.of(-1);
    private static final Overflow OVERFLOW = new Overflow();
    private static final IllegalArgumentException NOT_INT32 = new IllegalArgumentException("not a signed 32-bit value");

    private StarlarkInt() {
    }

    public static StarlarkInt of(int x) {
        int index = x - -128;
        if (0 <= index && index < smallints.length) {
            Int32 xi = smallints[index];
            if (xi == null) {
                StarlarkInt.smallints[index] = xi = new Int32(x);
            }
            return xi;
        }
        return new Int32(x);
    }

    public static StarlarkInt of(long x) {
        if ((long)((int)x) == x) {
            return StarlarkInt.of((int)x);
        }
        return new Int64(x);
    }

    public static StarlarkInt of(BigInteger x) {
        if (x.bitLength() < 64) {
            return StarlarkInt.of(x.longValue());
        }
        return new Big(x);
    }

    static StarlarkInt ofFiniteDouble(double x) {
        return StarlarkFloat.finiteDoubleToIntExact(x);
    }

    public static StarlarkInt parse(String s2, int base) {
        StarlarkInt result;
        String stringForErrors = s2;
        if (s2.isEmpty()) {
            throw new NumberFormatException("empty string");
        }
        boolean isNegative = false;
        char c = s2.charAt(0);
        if (c == '+') {
            s2 = s2.substring(1);
        } else if (c == '-') {
            s2 = s2.substring(1);
            isNegative = true;
        }
        String digits = s2;
        if (s2.length() > 1 && s2.charAt(0) == '0') {
            int prefixBase = 0;
            c = s2.charAt(1);
            if (c == 'b' || c == 'B') {
                prefixBase = 2;
            } else if (c == 'o' || c == 'O') {
                prefixBase = 8;
            } else if (c == 'x' || c == 'X') {
                prefixBase = 16;
            }
            if (prefixBase != 0 && (base == 0 || base == prefixBase)) {
                base = prefixBase;
                digits = s2.substring(2);
            }
        }
        if (digits == s2 && base == 0) {
            if (s2.length() > 1 && s2.charAt(0) == '0') {
                throw new NumberFormatException("cannot infer base when string begins with a 0: " + Starlark.repr(stringForErrors));
            }
            base = 10;
        }
        if (base < 2 || base > 36) {
            throw new NumberFormatException(String.format("invalid base %d (want 2 <= base <= 36)", base));
        }
        if (digits.startsWith("+") || digits.startsWith("-")) {
            throw new NumberFormatException(String.format("invalid base-%d literal: %s", base, Starlark.repr(stringForErrors)));
        }
        try {
            result = StarlarkInt.of(Long.parseLong(digits, base));
        }
        catch (NumberFormatException unused1) {
            try {
                result = StarlarkInt.of(new BigInteger(digits, base));
            }
            catch (NumberFormatException unused2) {
                throw new NumberFormatException(String.format("invalid base-%d literal: %s", base, Starlark.repr(stringForErrors)));
            }
        }
        return isNegative ? StarlarkInt.uminus(result) : result;
    }

    public abstract Number toNumber();

    public abstract int signum();

    public String toString() {
        if (this instanceof Int32) {
            return Integer.toString(((Int32)this).v);
        }
        if (this instanceof Int64) {
            return Long.toString(((Int64)this).v);
        }
        return this.toBigInteger().toString();
    }

    @Override
    public abstract void repr(Printer var1);

    public int toInt(String what) throws EvalException {
        throw Starlark.errorf("got %s for %s, want value in signed 32-bit range", this, what);
    }

    public long toLong(String what) throws EvalException {
        throw Starlark.errorf("got %s for %s, want value in the signed 64-bit range", this, what);
    }

    protected long toLongFast() throws Overflow {
        throw OVERFLOW;
    }

    public double toDouble() {
        if (this instanceof Int32) {
            return ((Int32)this).v;
        }
        if (this instanceof Int64) {
            return ((Int64)this).v;
        }
        return this.toBigInteger().doubleValue();
    }

    public double toFiniteDouble() throws EvalException {
        double d = this.toDouble();
        if (!Double.isFinite(d)) {
            throw Starlark.errorf("int too large to convert to float", new Object[0]);
        }
        return d;
    }

    public abstract BigInteger toBigInteger();

    public final int toIntUnchecked() throws IllegalArgumentException {
        if (this instanceof Int32) {
            return ((Int32)this).v;
        }
        throw NOT_INT32;
    }

    public int truncateToInt() {
        if (this instanceof Int32) {
            return ((Int32)this).v;
        }
        if (this instanceof Int64) {
            return (int)((Int64)this).v;
        }
        return this.toBigInteger().intValue();
    }

    @Override
    public boolean isImmutable() {
        return true;
    }

    @Override
    public boolean truth() {
        return this != ZERO;
    }

    @Override
    public int compareTo(StarlarkInt x) {
        return StarlarkInt.compare(this, x);
    }

    public static int compare(StarlarkInt x, StarlarkInt y) {
        try {
            long xl = x.toLongFast();
            try {
                long yl = y.toLongFast();
                return Long.compare(xl, yl);
            }
            catch (Overflow unused) {
                return -((Big)y).v.signum();
            }
        }
        catch (Overflow unused) {
            return y instanceof Big ? ((Big)x).v.compareTo(((Big)y).v) : ((Big)x).v.signum();
        }
    }

    public static StarlarkInt add(StarlarkInt x, StarlarkInt y) {
        if (x instanceof Int32 && y instanceof Int32) {
            long xl = ((Int32)x).v;
            long yl = ((Int32)y).v;
            return StarlarkInt.of(xl + yl);
        }
        try {
            boolean overflow;
            long xl = x.toLongFast();
            long yl = y.toLongFast();
            long zl = xl + yl;
            boolean bl = overflow = ((xl ^ zl) & (yl ^ zl)) < 0L;
            if (!overflow) {
                return StarlarkInt.of(zl);
            }
        }
        catch (Overflow xl) {
            // empty catch block
        }
        BigInteger xbig = x.toBigInteger();
        BigInteger ybig = y.toBigInteger();
        BigInteger zbig = xbig.add(ybig);
        return StarlarkInt.of(zbig);
    }

    public static StarlarkInt subtract(StarlarkInt x, StarlarkInt y) {
        if (x instanceof Int32 && y instanceof Int32) {
            long xl = ((Int32)x).v;
            long yl = ((Int32)y).v;
            return StarlarkInt.of(xl - yl);
        }
        try {
            boolean overflow;
            long xl = x.toLongFast();
            long yl = y.toLongFast();
            long zl = xl - yl;
            boolean bl = overflow = ((xl ^ yl) & (xl ^ zl)) < 0L;
            if (!overflow) {
                return StarlarkInt.of(zl);
            }
        }
        catch (Overflow xl) {
            // empty catch block
        }
        BigInteger xbig = x.toBigInteger();
        BigInteger ybig = y.toBigInteger();
        BigInteger zbig = xbig.subtract(ybig);
        return StarlarkInt.of(zbig);
    }

    public static StarlarkInt multiply(StarlarkInt x, StarlarkInt y) {
        BigInteger ybig;
        if (x instanceof Int32 && y instanceof Int32) {
            long xl = ((Int32)x).v;
            long yl = ((Int32)y).v;
            return StarlarkInt.of(xl * yl);
        }
        try {
            long xl = x.toLongFast();
            long yl = y.toLongFast();
            long xlo = xl & 0xFFFFFFFFL;
            long xhi = xl >> 32;
            long ylo = yl & 0xFFFFFFFFL;
            long yhi = yl >> 32;
            long zlo = xlo * ylo;
            long t2 = xhi * ylo + (zlo >>> 32);
            long z1 = t2 & 0xFFFFFFFFL;
            long z2 = t2 >> 32;
            long z128hi = xhi * yhi + z2 + ((z1 += xlo * yhi) >> 32);
            long z128lo = xl * yl;
            if (z128hi == (z128lo & Long.MIN_VALUE) >> 63) {
                return StarlarkInt.of(z128lo);
            }
        }
        catch (Overflow xl) {
            // empty catch block
        }
        if (x == ZERO || y == ONE) {
            return x;
        }
        if (y == ZERO || x == ONE) {
            return y;
        }
        if (x == MINUS_ONE) {
            return StarlarkInt.uminus(y);
        }
        if (y == MINUS_ONE) {
            return StarlarkInt.uminus(x);
        }
        BigInteger xbig = x.toBigInteger();
        BigInteger zbig = xbig.multiply(ybig = y.toBigInteger());
        StarlarkInt z = StarlarkInt.of(zbig);
        if (!(z instanceof Big)) {
            throw new AssertionError((Object)String.format("bug in multiplication: %s * %s = %s, must be long multiplication", x, y, z));
        }
        return z;
    }

    public static StarlarkInt floordiv(StarlarkInt x, StarlarkInt y) throws EvalException {
        if (y == ZERO) {
            throw Starlark.errorf("integer division by zero", new Object[0]);
        }
        try {
            long xl = x.toLongFast();
            long yl = y.toLongFast();
            if (xl != Long.MIN_VALUE || yl != -1L) {
                long quo = Math.floorDiv(xl, yl);
                return StarlarkInt.of(quo);
            }
        }
        catch (Overflow xl) {
            // empty catch block
        }
        BigInteger xbig = x.toBigInteger();
        BigInteger ybig = y.toBigInteger();
        BigInteger[] quorem = xbig.divideAndRemainder(ybig);
        if (xbig.signum() < 0 != ybig.signum() < 0 && quorem[1].signum() != 0) {
            quorem[0] = quorem[0].subtract(BigInteger.ONE);
        }
        return StarlarkInt.of(quorem[0]);
    }

    public static StarlarkInt mod(StarlarkInt x, StarlarkInt y) throws EvalException {
        if (y == ZERO) {
            throw Starlark.errorf("integer modulo by zero", new Object[0]);
        }
        try {
            long xl = x.toLongFast();
            long yl = y.toLongFast();
            return StarlarkInt.of(Math.floorMod(xl, yl));
        }
        catch (Overflow xl) {
            BigInteger xbig = x.toBigInteger();
            BigInteger ybig = y.toBigInteger();
            BigInteger zbig = xbig.remainder(ybig);
            if (x.signum() < 0 != y.signum() < 0 && zbig.signum() != 0) {
                zbig = zbig.add(ybig);
            }
            return StarlarkInt.of(zbig);
        }
    }

    public static StarlarkInt shiftRight(StarlarkInt x, StarlarkInt y) throws EvalException {
        int yi = y.toInt("shift count");
        if (yi < 0) {
            throw Starlark.errorf("negative shift count: %d", yi);
        }
        try {
            long xl = x.toLongFast();
            if (yi >= 64) {
                return xl < 0L ? StarlarkInt.of(-1) : ZERO;
            }
            return StarlarkInt.of(xl >> yi);
        }
        catch (Overflow xl) {
            BigInteger xbig = x.toBigInteger();
            BigInteger zbig = xbig.shiftRight(yi);
            return StarlarkInt.of(zbig);
        }
    }

    public static StarlarkInt shiftLeft(StarlarkInt x, StarlarkInt y) throws EvalException {
        int yi = y.toInt("shift count");
        if (yi < 0) {
            throw Starlark.errorf("negative shift count: %d", yi);
        }
        if (yi >= 512) {
            throw Starlark.errorf("shift count too large: %d", yi);
        }
        try {
            long xl = x.toLongFast();
            long z = xl << yi;
            if (z >> yi == xl && yi < 64) {
                return StarlarkInt.of(z);
            }
        }
        catch (Overflow xl) {
            // empty catch block
        }
        BigInteger xbig = x.toBigInteger();
        BigInteger zbig = xbig.shiftLeft(yi);
        return StarlarkInt.of(zbig);
    }

    public static StarlarkInt xor(StarlarkInt x, StarlarkInt y) {
        try {
            long xl = x.toLongFast();
            long yl = y.toLongFast();
            return StarlarkInt.of(xl ^ yl);
        }
        catch (Overflow xl) {
            BigInteger xbig = x.toBigInteger();
            BigInteger ybig = y.toBigInteger();
            BigInteger zbig = xbig.xor(ybig);
            return StarlarkInt.of(zbig);
        }
    }

    public static StarlarkInt or(StarlarkInt x, StarlarkInt y) {
        try {
            long xl = x.toLongFast();
            long yl = y.toLongFast();
            return StarlarkInt.of(xl | yl);
        }
        catch (Overflow xl) {
            BigInteger xbig = x.toBigInteger();
            BigInteger ybig = y.toBigInteger();
            BigInteger zbig = xbig.or(ybig);
            return StarlarkInt.of(zbig);
        }
    }

    public static StarlarkInt and(StarlarkInt x, StarlarkInt y) {
        try {
            long xl = x.toLongFast();
            long yl = y.toLongFast();
            return StarlarkInt.of(xl & yl);
        }
        catch (Overflow xl) {
            BigInteger xbig = x.toBigInteger();
            BigInteger ybig = y.toBigInteger();
            BigInteger zbig = xbig.and(ybig);
            return StarlarkInt.of(zbig);
        }
    }

    public static StarlarkInt bitnot(StarlarkInt x) {
        try {
            long xl = x.toLongFast();
            return StarlarkInt.of(xl ^ 0xFFFFFFFFFFFFFFFFL);
        }
        catch (Overflow xl) {
            BigInteger xbig = ((Big)x).v;
            return StarlarkInt.of(xbig.not());
        }
    }

    public static StarlarkInt uminus(StarlarkInt x) {
        long xl;
        if (x instanceof Int32) {
            long xl2 = ((Int32)x).v;
            return StarlarkInt.of(-xl2);
        }
        if (x instanceof Int64 && (xl = ((Int64)x).v) != Long.MIN_VALUE) {
            return StarlarkInt.of(-xl);
        }
        BigInteger xbig = x.toBigInteger();
        BigInteger ybig = xbig.negate();
        return StarlarkInt.of(ybig);
    }

    static boolean intEqualsFloat(StarlarkInt x, StarlarkFloat y) {
        double yf = y.toDouble();
        return !Double.isNaN(yf) && StarlarkInt.compareIntAndDouble(x, yf) == 0;
    }

    static int compareIntAndDouble(StarlarkInt x, double y) {
        int ysign;
        if (Double.isInfinite(y)) {
            return y > 0.0 ? -1 : 1;
        }
        if (x instanceof Int32 || x instanceof Int64 && StarlarkInt.longHasExactDouble(((Int64)x).v)) {
            double xf = x.toDouble();
            if (xf > y) {
                return 1;
            }
            if (xf < y) {
                return -1;
            }
            return 0;
        }
        int xsign = x.signum();
        if (xsign > (ysign = (int)Math.signum(y))) {
            return 1;
        }
        if (xsign < ysign) {
            return -1;
        }
        int shift = StarlarkFloat.getShift(y);
        BigInteger xbig = x.toBigInteger();
        if (shift < 0) {
            xbig = xbig.shiftLeft(-shift);
        }
        BigInteger ybig = BigInteger.valueOf(StarlarkFloat.getMantissa(y));
        if (shift > 0) {
            ybig = ybig.shiftLeft(shift);
        }
        return xbig.compareTo(ybig);
    }

    private static boolean longHasExactDouble(long x) {
        return (long)((double)x) == x;
    }

    private static final class Overflow
    extends Exception {
        private Overflow() {
        }
    }

    private static final class Big
    extends StarlarkInt {
        final BigInteger v;

        Big(BigInteger v) {
            this.v = v;
        }

        @Override
        public BigInteger toBigInteger() {
            return this.v;
        }

        @Override
        public Number toNumber() {
            return this.v;
        }

        @Override
        public int signum() {
            return this.v.signum();
        }

        @Override
        public void repr(Printer printer) {
            printer.append(this.v.toString());
        }

        public int hashCode() {
            return -292468197 * this.v.hashCode() ^ 0x6406918F;
        }

        public boolean equals(Object that) {
            return that instanceof Big && this.v.equals(((Big)that).v) || that instanceof StarlarkFloat && Big.intEqualsFloat(this, (StarlarkFloat)that);
        }
    }

    private static final class Int64
    extends StarlarkInt {
        final long v;

        Int64(long v) {
            this.v = v;
        }

        @Override
        public long toLong(String what) {
            return this.v;
        }

        @Override
        protected long toLongFast() {
            return this.v;
        }

        @Override
        public BigInteger toBigInteger() {
            return BigInteger.valueOf(this.v);
        }

        @Override
        public Number toNumber() {
            return this.v;
        }

        @Override
        public int signum() {
            return Long.signum(this.v);
        }

        @Override
        public void repr(Printer printer) {
            printer.append(this.v);
        }

        public int hashCode() {
            return 1740941269 * Long.hashCode(this.v) ^ 0xEE914A1B;
        }

        public boolean equals(Object that) {
            return that instanceof Int64 && this.v == ((Int64)that).v || that instanceof StarlarkFloat && Int64.intEqualsFloat(this, (StarlarkFloat)that);
        }
    }

    private static final class Int32
    extends StarlarkInt {
        final int v;

        Int32(int v) {
            this.v = v;
        }

        @Override
        public int toInt(String what) {
            return this.v;
        }

        @Override
        public long toLong(String what) {
            return this.v;
        }

        @Override
        protected long toLongFast() {
            return this.v;
        }

        @Override
        public BigInteger toBigInteger() {
            return BigInteger.valueOf(this.v);
        }

        @Override
        public Number toNumber() {
            return this.v;
        }

        @Override
        public int signum() {
            return Integer.signum(this.v);
        }

        @Override
        public void repr(Printer printer) {
            printer.append(this.v);
        }

        public int hashCode() {
            return 829182521 * Integer.hashCode(this.v) ^ 0x67C4A7D5;
        }

        public boolean equals(Object that) {
            return that instanceof Int32 && this.v == ((Int32)that).v || that instanceof StarlarkFloat && Int32.intEqualsFloat(this, (StarlarkFloat)that);
        }
    }
}

