/*
 * Decompiled with CFR 0.152.
 */
package net.starlark.java.eval;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Maps;
import com.google.errorprone.annotations.CanIgnoreReturnValue;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Set;
import java.util.function.BiConsumer;
import javax.annotation.Nullable;
import net.starlark.java.annot.Param;
import net.starlark.java.annot.StarlarkBuiltin;
import net.starlark.java.annot.StarlarkMethod;
import net.starlark.java.eval.EvalException;
import net.starlark.java.eval.Mutability;
import net.starlark.java.eval.Printer;
import net.starlark.java.eval.Starlark;
import net.starlark.java.eval.StarlarkIndexable;
import net.starlark.java.eval.StarlarkIterable;
import net.starlark.java.eval.StarlarkList;
import net.starlark.java.eval.StarlarkSemantics;
import net.starlark.java.eval.StarlarkThread;
import net.starlark.java.eval.StarlarkValue;
import net.starlark.java.eval.Tuple;

@StarlarkBuiltin(name="dict", category="core", doc="dict is a built-in type representing an associative mapping or <i>dictionary</i>. A dictionary supports indexing using <code>d[k]</code> and key membership testing using <code>k in d</code>; both operations take constant time. Unfrozen dictionaries are mutable, and may be updated by assigning to <code>d[k]</code> or by calling certain methods. Dictionaries are iterable; iteration yields the sequence of keys in insertion order. Iteration order is unaffected by updating the value associated with an existing key, but is affected by removing then reinserting a key.\n<pre>d = {0: 0, 2: 2, 1: 1}\n[k for k in d]  # [0, 2, 1]\nd.pop(2)\nd[0], d[2] = \"a\", \"b\"\n0 in d, \"a\" in d  # (True, False)\n[(k, v) for k, v in d.items()]  # [(0, \"a\"), (1, 1), (2, \"b\")]\n</pre>\n<p>There are four ways to construct a dictionary:\n<ol>\n<li>A dictionary expression <code>{k: v, ...}</code> yields a new dictionary with the specified key/value entries, inserted in the order they appear in the expression. Evaluation fails if any two key expressions yield the same value.\n<li>A dictionary comprehension <code>{k: v for vars in seq}</code> yields a new dictionary into which each key/value pair is inserted in loop iteration order. Duplicates are permitted: the first insertion of a given key determines its position in the sequence, and the last determines its associated value.\n<pre class=\"language-python\">\n{k: v for k, v in ((\"a\", 0), (\"b\", 1), (\"a\", 2))}  # {\"a\": 2, \"b\": 1}\n{i: 2*i for i in range(3)}  # {0: 0, 1: 2, 2: 4}\n</pre>\n<li>A call to the built-in <a href=\"../globals/all.html#dict\">dict</a> function returns a dictionary containing the specified entries, which are inserted in argument order, positional arguments before named. As with comprehensions, duplicate keys are permitted.\n<li>The union expression <code>x | y</code> yields a new dictionary by combining two existing dictionaries. If the two dictionaries have a key <code>k</code> in common, the right hand side dictionary's value of the key (in other words, <code>y[k]</code>) wins. The <code>|=</code> variant of the union operator modifies a dictionary in-place. Example:<br><pre class=language-python>d = {\"foo\": \"FOO\", \"bar\": \"BAR\"} | {\"foo\": \"FOO2\", \"baz\": \"BAZ\"}\n# d == {\"foo\": \"FOO2\", \"bar\": \"BAR\", \"baz\": \"BAZ\"}\nd = {\"a\": 1, \"b\": 2}\nd |= {\"b\": 3, \"c\": 4}\n# d == {\"a\": 1, \"b\": 3, \"c\": 4}</pre></ol>")
public class Dict<K, V>
implements Map<K, V>,
StarlarkValue,
Mutability.Freezable,
StarlarkIndexable,
StarlarkIterable<K> {
    private final Map<K, V> contents;
    private int iteratorCount;
    private Mutability mutability;
    private static final Dict<?, ?> EMPTY = new Dict(ImmutableMap.of());

    private Dict(Mutability mutability, LinkedHashMap<K, V> contents) {
        Preconditions.checkNotNull(mutability);
        Preconditions.checkState(mutability != Mutability.IMMUTABLE);
        this.mutability = mutability;
        this.contents = contents;
    }

    private Dict(ImmutableMap<K, V> contents) {
        this.mutability = Mutability.IMMUTABLE;
        this.contents = contents;
    }

    static <K, V> Dict<K, V> wrap(@Nullable Mutability mu, LinkedHashMap<K, V> contents) {
        if (mu == null) {
            mu = Mutability.IMMUTABLE;
        }
        if (mu == Mutability.IMMUTABLE && contents.isEmpty()) {
            return Dict.empty();
        }
        return new Dict<K, V>(mu, contents);
    }

    @Override
    public boolean truth() {
        return !this.isEmpty();
    }

    @Override
    public boolean isImmutable() {
        return this.mutability().isFrozen();
    }

    @Override
    public boolean updateIteratorCount(int delta) {
        if (this.mutability().isFrozen()) {
            return false;
        }
        if (delta > 0) {
            ++this.iteratorCount;
        } else if (delta < 0) {
            --this.iteratorCount;
        }
        return this.iteratorCount > 0;
    }

    @Override
    public void checkHashable() throws EvalException {
        throw Starlark.errorf("unhashable type: 'dict'", new Object[0]);
    }

    @Override
    public int hashCode() {
        return this.contents.hashCode();
    }

    @Override
    public boolean equals(Object o) {
        return this.contents.equals(o);
    }

    @Override
    public Iterator<K> iterator() {
        return this.keySet().iterator();
    }

    @StarlarkMethod(name="get", doc="Returns the value for <code>key</code> if <code>key</code> is in the dictionary, else <code>default</code>. If <code>default</code> is not given, it defaults to <code>None</code>, so that this method never throws an error.", parameters={@Param(name="key", doc="The key to look for."), @Param(name="default", defaultValue="None", named=true, doc="The default value to use (instead of None) if the key is not found.")}, useStarlarkThread=true)
    public Object get2(Object key, Object defaultValue, StarlarkThread thread) throws EvalException {
        V v = this.get(key);
        if (v != null) {
            return v;
        }
        this.containsKey(thread.getSemantics(), key);
        return defaultValue;
    }

    @StarlarkMethod(name="pop", doc="Removes a <code>key</code> from the dict, and returns the associated value. If no entry with that key was found, remove nothing and return the specified <code>default</code> value; if no default value was specified, fail instead.", parameters={@Param(name="key", doc="The key."), @Param(name="default", defaultValue="unbound", named=true, doc="a default value if the key is absent.")}, useStarlarkThread=true)
    public Object pop(Object key, Object defaultValue, StarlarkThread thread) throws EvalException {
        Starlark.checkMutable(this);
        V value = this.contents.remove(key);
        if (value != null) {
            return value;
        }
        Starlark.checkHashable(key);
        if (defaultValue != Starlark.UNBOUND) {
            return defaultValue;
        }
        throw Starlark.errorf("KeyError: %s", Starlark.repr(key));
    }

    @StarlarkMethod(name="popitem", doc="Remove and return the first <code>(key, value)</code> pair from the dictionary. <code>popitem</code> is useful to destructively iterate over a dictionary, as often used in set algorithms. If the dictionary is empty, the <code>popitem</code> call fails.")
    public Tuple popitem() throws EvalException {
        if (this.isEmpty()) {
            throw Starlark.errorf("popitem: empty dictionary", new Object[0]);
        }
        Starlark.checkMutable(this);
        Iterator<Map.Entry<K, V>> iterator = this.contents.entrySet().iterator();
        Map.Entry<K, V> entry = iterator.next();
        iterator.remove();
        return Tuple.pair(entry.getKey(), entry.getValue());
    }

    @StarlarkMethod(name="setdefault", doc="If <code>key</code> is in the dictionary, return its value. If not, insert key with a value of <code>default</code> and return <code>default</code>. <code>default</code> defaults to <code>None</code>.", parameters={@Param(name="key", doc="The key."), @Param(name="default", defaultValue="None", named=true, doc="a default value if the key is absent.")})
    public V setdefault(K key, V defaultValue) throws EvalException {
        Starlark.checkMutable(this);
        Starlark.checkHashable(key);
        V prev = this.contents.putIfAbsent(key, defaultValue);
        return prev != null ? prev : defaultValue;
    }

    @StarlarkMethod(name="update", doc="Updates the dictionary first with the optional positional argument, <code>pairs</code>,  then with the optional keyword arguments\nIf the positional argument is present, it must be a dict, iterable, or None.\nIf it is a dict, then its key/value pairs are inserted into this dict. If it is an iterable, it must provide a sequence of pairs (or other iterables of length 2), each of which is treated as a key/value pair to be inserted.\nEach keyword argument <code>name=value</code> causes the name/value pair to be inserted into this dict.", parameters={@Param(name="pairs", defaultValue="[]", doc="Either a dictionary or a list of entries. Entries must be tuples or lists with exactly two elements: key, value.")}, extraKeywords=@Param(name="kwargs", doc="Dictionary of additional entries."), useStarlarkThread=true)
    public void update(Object pairs, Dict<String, Object> kwargs, StarlarkThread thread) throws EvalException {
        Starlark.checkMutable(this);
        Dict dict = this;
        Dict.update("update", dict, pairs, kwargs);
    }

    static void update(String funcname, Dict<Object, Object> dict, Object pairs, Map<String, Object> kwargs) throws EvalException {
        if (pairs instanceof Map) {
            dict.putEntries((Map)pairs);
        } else {
            Iterable<?> iterable;
            try {
                iterable = Starlark.toIterable(pairs);
            }
            catch (EvalException unused) {
                throw Starlark.errorf("in %s, got %s, want iterable", funcname, Starlark.type(pairs));
            }
            int pos = 0;
            for (Object item : iterable) {
                Object[] pair;
                try {
                    pair = Starlark.toArray(item);
                }
                catch (EvalException unused) {
                    throw Starlark.errorf("in %s, dictionary update sequence element #%d is not iterable (%s)", funcname, pos, Starlark.type(item));
                }
                if (pair.length != 2) {
                    throw Starlark.errorf("in %s, item #%d has length %d, but exactly two elements are required", funcname, pos, pair.length);
                }
                dict.putEntry(pair[0], pair[1]);
                ++pos;
            }
        }
        dict.putEntries(kwargs);
    }

    @StarlarkMethod(name="values", doc="Returns the list of values:<pre class=\"language-python\">{2: \"a\", 4: \"b\", 1: \"c\"}.values() == [\"a\", \"b\", \"c\"]</pre>\n", useStarlarkThread=true)
    public StarlarkList<?> values0(StarlarkThread thread) throws EvalException {
        return StarlarkList.copyOf(thread.mutability(), this.values());
    }

    @StarlarkMethod(name="items", doc="Returns the list of key-value tuples:<pre class=\"language-python\">{2: \"a\", 4: \"b\", 1: \"c\"}.items() == [(2, \"a\"), (4, \"b\"), (1, \"c\")]</pre>\n", useStarlarkThread=true)
    public StarlarkList<?> items(StarlarkThread thread) throws EvalException {
        Object[] array = new Object[this.size()];
        int i = 0;
        for (Map.Entry<K, V> e : this.contents.entrySet()) {
            array[i++] = Tuple.pair(e.getKey(), e.getValue());
        }
        return StarlarkList.wrap(thread.mutability(), array);
    }

    @StarlarkMethod(name="keys", doc="Returns the list of keys:<pre class=\"language-python\">{2: \"a\", 4: \"b\", 1: \"c\"}.keys() == [2, 4, 1]</pre>\n", useStarlarkThread=true)
    public StarlarkList<?> keys(StarlarkThread thread) throws EvalException {
        Object[] array = new Object[this.size()];
        int i = 0;
        for (K e : this.contents.keySet()) {
            array[i++] = e;
        }
        return StarlarkList.wrap(thread.mutability(), array);
    }

    public static <K, V> Dict<K, V> empty() {
        return EMPTY;
    }

    public static <K, V> Dict<K, V> of(@Nullable Mutability mu) {
        if (mu == null) {
            mu = Mutability.IMMUTABLE;
        }
        if (mu == Mutability.IMMUTABLE) {
            return Dict.empty();
        }
        return new Dict(mu, new LinkedHashMap());
    }

    public static <K, V> Dict<K, V> copyOf(@Nullable Mutability mu, Map<? extends K, ? extends V> m4) {
        if (mu == null) {
            mu = Mutability.IMMUTABLE;
        }
        if (mu == Mutability.IMMUTABLE && m4 instanceof Dict && ((Dict)m4).isImmutable()) {
            Dict dict = (Dict)m4;
            return dict;
        }
        if (mu == Mutability.IMMUTABLE) {
            if (m4.isEmpty()) {
                return Dict.empty();
            }
            ImmutableMap.Builder<K, V> immutableMapBuilder = ImmutableMap.builderWithExpectedSize(m4.size());
            for (Map.Entry<K, V> e : m4.entrySet()) {
                immutableMapBuilder.put(Starlark.checkValid(e.getKey()), Starlark.checkValid(e.getValue()));
            }
            return new Dict(immutableMapBuilder.buildOrThrow());
        }
        LinkedHashMap<K, V> linkedHashMap = new LinkedHashMap<K, V>();
        for (Map.Entry<K, V> e : m4.entrySet()) {
            linkedHashMap.put(Starlark.checkValid(e.getKey()), Starlark.checkValid(e.getValue()));
        }
        return new Dict(mu, linkedHashMap);
    }

    public static <K, V> Dict<K, V> immutableCopyOf(Map<? extends K, ? extends V> m4) {
        return Dict.copyOf(null, m4);
    }

    public static <K, V> Builder<K, V> builder() {
        return new Builder();
    }

    @Override
    public Mutability mutability() {
        return this.mutability;
    }

    @Override
    public void unsafeShallowFreeze() {
        Mutability.Freezable.checkUnsafeShallowFreezePrecondition(this);
        this.mutability = Mutability.IMMUTABLE;
    }

    public void putEntry(K key, V value) throws EvalException {
        Starlark.checkMutable(this);
        Starlark.checkHashable(key);
        this.contents.put(key, value);
    }

    public <K2 extends K, V2 extends V> void putEntries(Map<K2, V2> map) throws EvalException {
        Starlark.checkMutable(this);
        for (Map.Entry<K2, V2> e : map.entrySet()) {
            K2 k = e.getKey();
            Starlark.checkHashable(k);
            this.contents.put(k, e.getValue());
        }
    }

    V removeEntry(Object key) throws EvalException {
        Starlark.checkMutable(this);
        return this.contents.remove(key);
    }

    @StarlarkMethod(name="clear", doc="Remove all items from the dictionary.")
    public void clearEntries() throws EvalException {
        Starlark.checkMutable(this);
        this.contents.clear();
    }

    @Override
    public void repr(Printer printer) {
        printer.printList(this.entrySet(), "{", ", ", "}");
    }

    public String toString() {
        return Starlark.repr(this);
    }

    public static <K, V> Dict<K, V> cast(Object x, Class<K> keyType, Class<V> valueType, String what) throws EvalException {
        Preconditions.checkNotNull(x);
        if (!(x instanceof Dict)) {
            throw Starlark.errorf("got %s for '%s', want dict", Starlark.type(x), what);
        }
        for (Map.Entry e : ((Map)x).entrySet()) {
            if (keyType.isAssignableFrom(e.getKey().getClass()) && valueType.isAssignableFrom(e.getValue().getClass())) continue;
            throw Starlark.errorf("got dict<%s, %s> for '%s', want dict<%s, %s>", Starlark.type(e.getKey()), Starlark.type(e.getValue()), what, Starlark.classType(keyType), Starlark.classType(valueType));
        }
        Dict res = (Dict)x;
        return res;
    }

    public static <K, V> Dict<K, V> noneableCast(Object x, Class<K> keyType, Class<V> valueType, String what) throws EvalException {
        return x == Starlark.NONE ? Dict.empty() : Dict.cast(x, keyType, valueType, what);
    }

    @Override
    public Object getIndex(StarlarkSemantics semantics, Object key) throws EvalException {
        V v = this.get(key);
        if (v == null) {
            throw Starlark.errorf("key %s not found in dictionary", Starlark.repr(key));
        }
        return v;
    }

    @Override
    public boolean containsKey(StarlarkSemantics semantics, Object key) throws EvalException {
        Starlark.checkHashable(key);
        return this.containsKey(key);
    }

    @Override
    public boolean containsKey(Object key) {
        return this.contents.containsKey(key);
    }

    @Override
    public boolean containsValue(Object value) {
        return this.contents.containsValue(value);
    }

    @Override
    public Set<Map.Entry<K, V>> entrySet() {
        return Collections.unmodifiableMap(this.contents).entrySet();
    }

    @Override
    @Nullable
    public V get(Object key) {
        return this.contents.get(key);
    }

    @Override
    public boolean isEmpty() {
        return this.contents.isEmpty();
    }

    @Override
    public Set<K> keySet() {
        return Collections.unmodifiableMap(this.contents).keySet();
    }

    @Override
    public int size() {
        return this.contents.size();
    }

    @Override
    public Collection<V> values() {
        return Collections.unmodifiableMap(this.contents).values();
    }

    @Override
    @Deprecated
    public void clear() {
        throw new UnsupportedOperationException();
    }

    @Override
    @Nullable
    @Deprecated
    public V put(K key, V value) {
        throw new UnsupportedOperationException();
    }

    @Override
    @Deprecated
    public void putAll(Map<? extends K, ? extends V> map) {
        throw new UnsupportedOperationException();
    }

    @Override
    @Nullable
    @Deprecated
    public V remove(Object key) {
        throw new UnsupportedOperationException();
    }

    public static final class ImmutableKeyTrackingDict<K, V>
    extends Dict<K, V> {
        private final ImmutableSet.Builder<K> accessedKeys = ImmutableSet.builder();

        private ImmutableKeyTrackingDict(ImmutableMap<K, V> contents) {
            super(contents);
        }

        public ImmutableSet<K> getAccessedKeys() {
            return this.accessedKeys.build();
        }

        @Override
        public boolean containsKey(Object key) {
            if (super.containsKey(key)) {
                this.accessedKeys.add(key);
                return true;
            }
            return false;
        }

        @Override
        @Nullable
        public V get(Object key) {
            Object value = super.get(key);
            if (value != null) {
                this.accessedKeys.add(key);
            }
            return value;
        }

        @Override
        public Set<K> keySet() {
            Set keySet = super.keySet();
            this.accessedKeys.addAll((Iterable)keySet);
            return keySet;
        }

        @Override
        public Set<Map.Entry<K, V>> entrySet() {
            this.accessedKeys.addAll((Iterable)super.keySet());
            return super.entrySet();
        }
    }

    public static final class Builder<K, V> {
        private final ArrayList<Object> items = new ArrayList();

        @CanIgnoreReturnValue
        public Builder<K, V> put(K k, V v) {
            this.items.add(Starlark.checkValid(k));
            this.items.add(Starlark.checkValid(v));
            return this;
        }

        @CanIgnoreReturnValue
        public Builder<K, V> putAll(Map<? extends K, ? extends V> map) {
            this.items.ensureCapacity(this.items.size() + 2 * map.size());
            for (Map.Entry<K, V> e : map.entrySet()) {
                this.put(e.getKey(), e.getValue());
            }
            return this;
        }

        public Dict<K, V> buildImmutable() {
            return this.build(null);
        }

        public ImmutableKeyTrackingDict<K, V> buildImmutableWithKeyTracking() {
            return new ImmutableKeyTrackingDict<K, V>(this.buildImmutableMap());
        }

        public Dict<K, V> build(@Nullable Mutability mu) {
            if (mu == null) {
                mu = Mutability.IMMUTABLE;
            }
            if (mu == Mutability.IMMUTABLE) {
                if (this.items.isEmpty()) {
                    return Dict.empty();
                }
                return new Dict<K, V>(this.buildImmutableMap());
            }
            return new Dict<K, V>(mu, this.buildLinkedHashMap());
        }

        private void populateMap(int n, BiConsumer<K, V> mapEntryConsumer) {
            for (int i = 0; i < n; ++i) {
                Object k = this.items.get(2 * i);
                Object v = this.items.get(2 * i + 1);
                mapEntryConsumer.accept(k, v);
            }
        }

        private ImmutableMap<K, V> buildImmutableMap() {
            int n = this.items.size() / 2;
            ImmutableMap.Builder immutableMapBuilder = ImmutableMap.builderWithExpectedSize(n);
            this.populateMap(n, immutableMapBuilder::put);
            return immutableMapBuilder.buildKeepingLast();
        }

        private LinkedHashMap<K, V> buildLinkedHashMap() {
            int n = this.items.size() / 2;
            LinkedHashMap map = Maps.newLinkedHashMapWithExpectedSize(n);
            this.populateMap(n, map::put);
            return map;
        }
    }
}

